"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SketchesServiceImpl = void 0;
const inversify_1 = require("inversify");
const minimatch = require("minimatch");
const fs = require("fs");
const os = require("os");
const temp = require("temp");
const path = require("path");
const crypto = require("crypto");
const ncp_1 = require("ncp");
const util_1 = require("util");
const uri_1 = require("@theia/core/lib/common/uri");
const node_1 = require("@theia/core/lib/node");
const os_1 = require("@theia/core/lib/common/os");
const config_service_1 = require("../common/protocol/config-service");
const sketches_service_1 = require("../common/protocol/sketches-service");
const utils_1 = require("../common/utils");
const notification_service_server_1 = require("./notification-service-server");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const core_client_provider_1 = require("./core-client-provider");
const commands_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/commands_pb");
const WIN32_DRIVE_REGEXP = /^[a-zA-Z]:\\/;
const prefix = '.arduinoIDE-unsaved';
let SketchesServiceImpl = class SketchesServiceImpl extends core_client_provider_1.CoreClientAware {
    constructor() {
        super(...arguments);
        this.sketchSuffixIndex = 1;
    }
    async getSketches({ uri, exclude, }) {
        const start = Date.now();
        let sketchbookPath;
        if (!uri) {
            const { sketchDirUri } = await this.configService.getConfiguration();
            sketchbookPath = node_1.FileUri.fsPath(sketchDirUri);
            if (!(await util_1.promisify(fs.exists)(sketchbookPath))) {
                await util_1.promisify(fs.mkdir)(sketchbookPath, { recursive: true });
            }
        }
        else {
            sketchbookPath = node_1.FileUri.fsPath(uri);
        }
        const container = {
            label: uri ? path.basename(sketchbookPath) : 'Sketchbook',
            sketches: [],
            children: [],
        };
        if (!(await util_1.promisify(fs.exists)(sketchbookPath))) {
            return container;
        }
        const stat = await util_1.promisify(fs.stat)(sketchbookPath);
        if (!stat.isDirectory()) {
            return container;
        }
        const recursivelyLoad = async (fsPath, containerToLoad) => {
            const filenames = await util_1.promisify(fs.readdir)(fsPath);
            for (const name of filenames) {
                const childFsPath = path.join(fsPath, name);
                let skip = false;
                for (const pattern of exclude || [
                    '**/libraries/**',
                    '**/hardware/**',
                ]) {
                    if (!skip && minimatch(childFsPath, pattern)) {
                        skip = true;
                    }
                }
                if (skip) {
                    continue;
                }
                try {
                    const stat = await util_1.promisify(fs.stat)(childFsPath);
                    if (stat.isDirectory()) {
                        const sketch = await this._isSketchFolder(node_1.FileUri.create(childFsPath).toString());
                        if (sketch) {
                            containerToLoad.sketches.push(Object.assign(Object.assign({}, sketch), { mtimeMs: stat.mtimeMs }));
                        }
                        else {
                            const childContainer = {
                                label: name,
                                children: [],
                                sketches: [],
                            };
                            await recursivelyLoad(childFsPath, childContainer);
                            if (!sketches_service_1.SketchContainer.isEmpty(childContainer)) {
                                containerToLoad.children.push(childContainer);
                            }
                        }
                    }
                }
                catch (_a) {
                    console.warn(`Could not load sketch from ${childFsPath}.`);
                }
            }
            containerToLoad.sketches.sort((left, right) => right.mtimeMs - left.mtimeMs);
            return containerToLoad;
        };
        await recursivelyLoad(sketchbookPath, container);
        sketches_service_1.SketchContainer.prune(container);
        console.debug(`Loading the sketches from ${sketchbookPath} took ${Date.now() - start} ms.`);
        return container;
    }
    async loadSketch(uri) {
        await this.coreClientProvider.initialized;
        const { client, instance } = await this.coreClient();
        const req = new commands_pb_1.LoadSketchRequest();
        req.setSketchPath(node_1.FileUri.fsPath(uri));
        req.setInstance(instance);
        const sketch = await new Promise((resolve, reject) => {
            client.loadSketch(req, async (err, resp) => {
                if (err) {
                    reject(err);
                    return;
                }
                const sketchFolderPath = resp.getLocationPath();
                const { mtimeMs } = await util_1.promisify(fs.lstat)(sketchFolderPath);
                resolve({
                    name: path.basename(sketchFolderPath),
                    uri: node_1.FileUri.create(sketchFolderPath).toString(),
                    mainFileUri: node_1.FileUri.create(resp.getMainFile()).toString(),
                    otherSketchFileUris: resp
                        .getOtherSketchFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    additionalFileUris: resp
                        .getAdditionalFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    rootFolderFileUris: resp
                        .getRootFolderFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    mtimeMs,
                });
            });
        });
        return sketch;
    }
    async maybeLoadSketch(uri) {
        return this._isSketchFolder(uri);
    }
    get recentSketchesFsPath() {
        return this.envVariableServer
            .getConfigDirUri()
            .then((uri) => path.join(node_1.FileUri.fsPath(uri), 'recent-sketches.json'));
    }
    async loadRecentSketches(fsPath) {
        let data = {};
        try {
            const raw = await util_1.promisify(fs.readFile)(fsPath, {
                encoding: 'utf8',
            });
            data = JSON.parse(raw);
        }
        catch (_a) { }
        return data;
    }
    async markAsRecentlyOpened(uri) {
        let sketch = undefined;
        try {
            sketch = await this.loadSketch(uri);
        }
        catch (_a) {
            return;
        }
        if (await this.isTemp(sketch)) {
            return;
        }
        const fsPath = await this.recentSketchesFsPath;
        const data = await this.loadRecentSketches(fsPath);
        const now = Date.now();
        data[sketch.uri] = now;
        let toDeleteUri = undefined;
        if (Object.keys(data).length > 10) {
            let min = Number.MAX_SAFE_INTEGER;
            for (const uri of Object.keys(data)) {
                if (min > data[uri]) {
                    min = data[uri];
                    toDeleteUri = uri;
                }
            }
        }
        if (toDeleteUri) {
            delete data[toDeleteUri];
        }
        await util_1.promisify(fs.writeFile)(fsPath, JSON.stringify(data, null, 2));
        this.recentlyOpenedSketches().then((sketches) => this.notificationService.notifyRecentSketchesChanged({ sketches }));
    }
    async recentlyOpenedSketches() {
        const configDirUri = await this.envVariableServer.getConfigDirUri();
        const fsPath = path.join(node_1.FileUri.fsPath(configDirUri), 'recent-sketches.json');
        let data = {};
        try {
            const raw = await util_1.promisify(fs.readFile)(fsPath, {
                encoding: 'utf8',
            });
            data = JSON.parse(raw);
        }
        catch (_a) { }
        const sketches = [];
        for (const uri of Object.keys(data).sort((left, right) => data[right] - data[left])) {
            try {
                const sketch = await this.loadSketch(uri);
                sketches.push(sketch);
            }
            catch (_b) { }
        }
        return sketches;
    }
    async cloneExample(uri) {
        const sketch = await this.loadSketch(uri);
        const parentPath = await new Promise((resolve, reject) => {
            temp.mkdir({ prefix }, (err, dirPath) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(dirPath);
            });
        });
        const destinationUri = node_1.FileUri.create(path.join(parentPath, sketch.name)).toString();
        const copiedSketchUri = await this.copy(sketch, { destinationUri });
        return this.loadSketch(copiedSketchUri);
    }
    async createNewSketch() {
        const monthNames = [
            'jan',
            'feb',
            'mar',
            'apr',
            'may',
            'jun',
            'jul',
            'aug',
            'sep',
            'oct',
            'nov',
            'dec',
        ];
        const today = new Date();
        const parentPath = await new Promise((resolve, reject) => {
            temp.mkdir({ prefix }, (err, dirPath) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(dirPath);
            });
        });
        const sketchBaseName = `sketch_${monthNames[today.getMonth()]}${today.getDate()}`;
        const config = await this.configService.getConfiguration();
        const sketchbookPath = node_1.FileUri.fsPath(config.sketchDirUri);
        let sketchName;
        // If it's another day, reset the count of sketches created today
        if (this.lastSketchBaseName !== sketchBaseName)
            this.sketchSuffixIndex = 1;
        let nameFound = false;
        while (!nameFound) {
            const sketchNameCandidate = `${sketchBaseName}${sketchIndexToLetters(this.sketchSuffixIndex++)}`;
            // Note: we check the future destination folder (`directories.user`) for name collision and not the temp folder!
            const sketchExists = await util_1.promisify(fs.exists)(path.join(sketchbookPath, sketchNameCandidate));
            if (!sketchExists) {
                nameFound = true;
                sketchName = sketchNameCandidate;
            }
        }
        if (!sketchName) {
            throw new Error('Cannot create a unique sketch name');
        }
        this.lastSketchBaseName = sketchBaseName;
        const sketchDir = path.join(parentPath, sketchName);
        const sketchFile = path.join(sketchDir, `${sketchName}.ino`);
        await util_1.promisify(fs.mkdir)(sketchDir, { recursive: true });
        await util_1.promisify(fs.writeFile)(sketchFile, `void setup() {
  // put your setup code here, to run once:

}

void loop() {
  // put your main code here, to run repeatedly:

}
`, { encoding: 'utf8' });
        return this.loadSketch(node_1.FileUri.create(sketchDir).toString());
    }
    async getSketchFolder(uri) {
        if (!uri) {
            return undefined;
        }
        let currentUri = new uri_1.default(uri);
        while (currentUri && !currentUri.path.isRoot) {
            const sketch = await this._isSketchFolder(currentUri.toString());
            if (sketch) {
                return sketch;
            }
            currentUri = currentUri.parent;
        }
        return undefined;
    }
    async isSketchFolder(uri) {
        const sketch = await this._isSketchFolder(uri);
        return !!sketch;
    }
    async _isSketchFolder(uri) {
        const fsPath = node_1.FileUri.fsPath(uri);
        let stat;
        try {
            stat = await util_1.promisify(fs.lstat)(fsPath);
        }
        catch (_a) { }
        if (stat && stat.isDirectory()) {
            const basename = path.basename(fsPath);
            const files = await util_1.promisify(fs.readdir)(fsPath);
            for (let i = 0; i < files.length; i++) {
                if (files[i] === basename + '.ino' || files[i] === basename + '.pde') {
                    try {
                        const sketch = await this.loadSketch(node_1.FileUri.create(fsPath).toString());
                        return sketch;
                    }
                    catch (_b) { }
                }
            }
        }
        return undefined;
    }
    async isTemp(sketch) {
        let sketchPath = node_1.FileUri.fsPath(sketch.uri);
        let temp = await util_1.promisify(fs.realpath)(os.tmpdir());
        // Note: VS Code URI normalizes the drive letter. `C:` will be converted into `c:`.
        // https://github.com/Microsoft/vscode/issues/68325#issuecomment-462239992
        if (os_1.isWindows) {
            if (WIN32_DRIVE_REGEXP.exec(sketchPath)) {
                sketchPath = utils_1.firstToLowerCase(sketchPath);
            }
            if (WIN32_DRIVE_REGEXP.exec(temp)) {
                temp = utils_1.firstToLowerCase(temp);
            }
        }
        return sketchPath.indexOf(prefix) !== -1 && sketchPath.startsWith(temp);
    }
    async copy(sketch, { destinationUri }) {
        const source = node_1.FileUri.fsPath(sketch.uri);
        const exists = await util_1.promisify(fs.exists)(source);
        if (!exists) {
            throw new Error(`Sketch does not exist: ${sketch}`);
        }
        // Nothing to do when source and destination are the same.
        if (sketch.uri === destinationUri) {
            await this.loadSketch(sketch.uri); // Sanity check.
            return sketch.uri;
        }
        const copy = async (sourcePath, destinationPath) => {
            return new Promise((resolve, reject) => {
                ncp_1.ncp.ncp(sourcePath, destinationPath, async (error) => {
                    if (error) {
                        reject(error);
                        return;
                    }
                    const newName = path.basename(destinationPath);
                    try {
                        const oldPath = path.join(destinationPath, new uri_1.default(sketch.mainFileUri).path.base);
                        const newPath = path.join(destinationPath, `${newName}.ino`);
                        if (oldPath !== newPath) {
                            await util_1.promisify(fs.rename)(oldPath, newPath);
                        }
                        await this.loadSketch(node_1.FileUri.create(destinationPath).toString()); // Sanity check.
                        resolve();
                    }
                    catch (e) {
                        reject(e);
                    }
                });
            });
        };
        // https://github.com/arduino/arduino-ide/issues/65
        // When copying `/path/to/sketchbook/sketch_A` to `/path/to/sketchbook/sketch_A/anything` on a non-POSIX filesystem,
        // `ncp` makes a recursion and copies the folders over and over again. In such cases, we copy the source into a temp folder,
        // then move it to the desired destination.
        const destination = node_1.FileUri.fsPath(destinationUri);
        let tempDestination = await new Promise((resolve, reject) => {
            temp.track().mkdir({ prefix }, async (err, dirPath) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(dirPath);
            });
        });
        tempDestination = path.join(tempDestination, sketch.name);
        await fs.promises.mkdir(tempDestination, { recursive: true });
        await copy(source, tempDestination);
        await copy(tempDestination, destination);
        return node_1.FileUri.create(destination).toString();
    }
    async archive(sketch, destinationUri) {
        await this.coreClientProvider.initialized;
        await this.loadSketch(sketch.uri); // sanity check
        const { client } = await this.coreClient();
        const archivePath = node_1.FileUri.fsPath(destinationUri);
        // The CLI cannot override existing archives, so we have to wipe it manually: https://github.com/arduino/arduino-cli/issues/1160
        if (await util_1.promisify(fs.exists)(archivePath)) {
            await util_1.promisify(fs.unlink)(archivePath);
        }
        const req = new commands_pb_1.ArchiveSketchRequest();
        req.setSketchPath(node_1.FileUri.fsPath(sketch.uri));
        req.setArchivePath(archivePath);
        await new Promise((resolve, reject) => {
            client.archiveSketch(req, (err) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(destinationUri);
            });
        });
        return destinationUri;
    }
    async getIdeTempFolderUri(sketch) {
        const genBuildPath = await this.getIdeTempFolderPath(sketch);
        return node_1.FileUri.create(genBuildPath).toString();
    }
    async getIdeTempFolderPath(sketch) {
        const sketchPath = node_1.FileUri.fsPath(sketch.uri);
        await fs.promises.readdir(sketchPath); // Validates the sketch folder and rejects if not accessible.
        const suffix = crypto.createHash('md5').update(sketchPath).digest('hex');
        return path.join(os.tmpdir(), `arduino-ide2-${suffix}`);
    }
};
__decorate([
    inversify_1.inject(config_service_1.ConfigService),
    __metadata("design:type", Object)
], SketchesServiceImpl.prototype, "configService", void 0);
__decorate([
    inversify_1.inject(notification_service_server_1.NotificationServiceServerImpl),
    __metadata("design:type", notification_service_server_1.NotificationServiceServerImpl)
], SketchesServiceImpl.prototype, "notificationService", void 0);
__decorate([
    inversify_1.inject(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], SketchesServiceImpl.prototype, "envVariableServer", void 0);
SketchesServiceImpl = __decorate([
    inversify_1.injectable()
], SketchesServiceImpl);
exports.SketchesServiceImpl = SketchesServiceImpl;
/*
 * When a new sketch is created, add a suffix to distinguish it
 * from other new sketches I created today.
 * If 'sketch_jul8a' is already used, go with 'sketch_jul8b'.
 * If 'sketch_jul8b' already used, go with 'sketch_jul8c'.
 * When it reacheas 'sketch_jul8z', go with 'sketch_jul8aa',
 * and so on.
 */
function sketchIndexToLetters(num) {
    let out = '';
    let pow;
    do {
        pow = Math.floor(num / 26);
        const mod = num % 26;
        out = (mod ? String.fromCharCode(96 + mod) : (--pow, 'z')) + out;
        num = pow;
    } while (pow > 0);
    return out;
}
//# sourceMappingURL=sketches-service-impl.js.map