"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SerialServiceImpl = exports.SerialServiceName = void 0;
const util_1 = require("util");
const inversify_1 = require("inversify");
const struct_pb_1 = require("google-protobuf/google/protobuf/struct_pb");
const logger_1 = require("@theia/core/lib/common/logger");
const serial_service_1 = require("../../common/protocol/serial-service");
const monitor_pb_1 = require("../cli-protocol/cc/arduino/cli/monitor/v1/monitor_pb");
const monitor_client_provider_1 = require("./monitor-client-provider");
const boards_service_1 = require("../../common/protocol/boards-service");
const web_socket_service_1 = require("../web-socket/web-socket-service");
const protocol_1 = require("../../browser/serial/plotter/protocol");
exports.SerialServiceName = 'serial-service';
var ErrorWithCode;
(function (ErrorWithCode) {
    function toSerialError(error, config) {
        const { message } = error;
        let code = undefined;
        if (is(error)) {
            // TODO: const `mapping`. Use regex for the `message`.
            const mapping = new Map();
            mapping.set('1 CANCELLED: Cancelled on client', serial_service_1.SerialError.ErrorCodes.CLIENT_CANCEL);
            mapping.set('2 UNKNOWN: device not configured', serial_service_1.SerialError.ErrorCodes.DEVICE_NOT_CONFIGURED);
            mapping.set('2 UNKNOWN: error opening serial connection: Serial port busy', serial_service_1.SerialError.ErrorCodes.DEVICE_BUSY);
            code = mapping.get(message);
        }
        return {
            message,
            code,
            config,
        };
    }
    ErrorWithCode.toSerialError = toSerialError;
    function is(error) {
        return typeof error.code === 'number';
    }
})(ErrorWithCode || (ErrorWithCode = {}));
let SerialServiceImpl = class SerialServiceImpl {
    constructor(logger, serialClientProvider, webSocketService) {
        this.logger = logger;
        this.serialClientProvider = serialClientProvider;
        this.webSocketService = webSocketService;
        this.messages = [];
        this.uploadInProgress = false;
    }
    async isSerialPortOpen() {
        return !!this.serialConnection;
    }
    setClient(client) {
        var _a;
        this.theiaFEClient = client;
        (_a = this.theiaFEClient) === null || _a === void 0 ? void 0 : _a.notifyWebSocketChanged(this.webSocketService.getAddress().port);
        // listen for the number of websocket clients and create or dispose the serial connection
        this.onWSClientsNumberChanged =
            this.webSocketService.onClientsNumberChanged(async () => {
                await this.connectSerialIfRequired();
            });
    }
    async clientsAttached() {
        return this.webSocketService.getConnectedClientsNumber.bind(this.webSocketService)();
    }
    async connectSerialIfRequired() {
        if (this.uploadInProgress)
            return;
        const clients = await this.clientsAttached();
        clients > 0 ? await this.connect() : await this.disconnect();
    }
    dispose() {
        this.logger.info('>>> Disposing serial service...');
        if (this.serialConnection) {
            this.disconnect();
        }
        this.logger.info('<<< Disposed serial service.');
        this.theiaFEClient = undefined;
    }
    async setSerialConfig(config) {
        this.serialConfig = config;
        await this.disconnect();
        await this.connectSerialIfRequired();
    }
    async updateWsConfigParam(config) {
        const msg = {
            command: protocol_1.SerialPlotter.Protocol.Command.MIDDLEWARE_CONFIG_CHANGED,
            data: config,
        };
        this.webSocketService.sendMessage(JSON.stringify(msg));
    }
    async connect() {
        if (!this.serialConfig) {
            return serial_service_1.Status.CONFIG_MISSING;
        }
        this.logger.info(`>>> Creating serial connection for ${boards_service_1.Board.toString(this.serialConfig.board)} on port ${this.serialConfig.port.address}...`);
        if (this.serialConnection) {
            return serial_service_1.Status.ALREADY_CONNECTED;
        }
        const client = await this.serialClientProvider.client();
        if (!client) {
            return serial_service_1.Status.NOT_CONNECTED;
        }
        if (client instanceof Error) {
            return { message: client.message };
        }
        const duplex = client.streamingOpen();
        this.serialConnection = { duplex, config: this.serialConfig };
        const serialConfig = this.serialConfig;
        duplex.on('error', ((error) => {
            const serialError = ErrorWithCode.toSerialError(error, serialConfig);
            if (serialError.code !== serial_service_1.SerialError.ErrorCodes.CLIENT_CANCEL) {
                this.disconnect(serialError).then(() => {
                    if (this.theiaFEClient) {
                        this.theiaFEClient.notifyError(serialError);
                    }
                });
            }
            if (serialError.code === undefined) {
                // Log the original, unexpected error.
                this.logger.error(error);
            }
        }).bind(this));
        this.updateWsConfigParam({ connected: !!this.serialConnection });
        const flushMessagesToFrontend = () => {
            if (this.messages.length) {
                this.webSocketService.sendMessage(JSON.stringify(this.messages));
                this.messages = [];
            }
        };
        this.onMessageReceived = this.webSocketService.onMessageReceived((msg) => {
            var _a, _b, _c;
            try {
                const message = JSON.parse(msg);
                switch (message.command) {
                    case protocol_1.SerialPlotter.Protocol.Command.PLOTTER_SEND_MESSAGE:
                        this.sendMessageToSerial(message.data);
                        break;
                    case protocol_1.SerialPlotter.Protocol.Command.PLOTTER_SET_BAUDRATE:
                        (_a = this.theiaFEClient) === null || _a === void 0 ? void 0 : _a.notifyBaudRateChanged(parseInt(message.data, 10));
                        break;
                    case protocol_1.SerialPlotter.Protocol.Command.PLOTTER_SET_LINE_ENDING:
                        (_b = this.theiaFEClient) === null || _b === void 0 ? void 0 : _b.notifyLineEndingChanged(message.data);
                        break;
                    case protocol_1.SerialPlotter.Protocol.Command.PLOTTER_SET_INTERPOLATE:
                        (_c = this.theiaFEClient) === null || _c === void 0 ? void 0 : _c.notifyInterpolateChanged(message.data);
                        break;
                    default:
                        break;
                }
            }
            catch (error) { }
        });
        // empty the queue every 32ms (~30fps)
        this.flushMessagesInterval = setInterval(flushMessagesToFrontend, 32);
        duplex.on('data', ((resp) => {
            const raw = resp.getData();
            const message = typeof raw === 'string' ? raw : new TextDecoder('utf8').decode(raw);
            // split the message if it contains more lines
            const messages = stringToArray(message);
            this.messages.push(...messages);
        }).bind(this));
        const { type, port } = this.serialConfig;
        const req = new monitor_pb_1.StreamingOpenRequest();
        const monitorConfig = new monitor_pb_1.MonitorConfig();
        monitorConfig.setType(this.mapType(type));
        monitorConfig.setTarget(port.address);
        if (this.serialConfig.baudRate !== undefined) {
            monitorConfig.setAdditionalConfig(struct_pb_1.Struct.fromJavaScript({ BaudRate: this.serialConfig.baudRate }));
        }
        req.setConfig(monitorConfig);
        if (!this.serialConnection) {
            return await this.disconnect();
        }
        const writeTimeout = new Promise((resolve) => {
            setTimeout(async () => {
                resolve(serial_service_1.Status.NOT_CONNECTED);
            }, 1000);
        });
        const writePromise = (serialConnection) => {
            return new Promise((resolve) => {
                serialConnection.duplex.write(req, () => {
                    var _a, _b;
                    const boardName = ((_a = this.serialConfig) === null || _a === void 0 ? void 0 : _a.board) ? boards_service_1.Board.toString(this.serialConfig.board, {
                        useFqbn: false,
                    })
                        : 'unknown board';
                    const portName = ((_b = this.serialConfig) === null || _b === void 0 ? void 0 : _b.port) ? this.serialConfig.port.address
                        : 'unknown port';
                    this.logger.info(`<<< Serial connection created for ${boardName} on port ${portName}.`);
                    resolve(serial_service_1.Status.OK);
                });
            });
        };
        const status = await Promise.race([
            writeTimeout,
            writePromise(this.serialConnection),
        ]);
        if (status === serial_service_1.Status.NOT_CONNECTED) {
            this.disconnect();
        }
        return status;
    }
    async disconnect(reason) {
        return new Promise((resolve) => {
            try {
                if (this.onMessageReceived) {
                    this.onMessageReceived.dispose();
                    this.onMessageReceived = null;
                }
                if (this.flushMessagesInterval) {
                    clearInterval(this.flushMessagesInterval);
                    this.flushMessagesInterval = null;
                }
                if (!this.serialConnection &&
                    reason &&
                    reason.code === serial_service_1.SerialError.ErrorCodes.CLIENT_CANCEL) {
                    resolve(serial_service_1.Status.OK);
                    return;
                }
                this.logger.info('>>> Disposing serial connection...');
                if (!this.serialConnection) {
                    this.logger.warn('<<< Not connected. Nothing to dispose.');
                    resolve(serial_service_1.Status.NOT_CONNECTED);
                    return;
                }
                const { duplex, config } = this.serialConnection;
                this.logger.info(`<<< Disposed serial connection for ${boards_service_1.Board.toString(config.board, {
                    useFqbn: false,
                })} on port ${config.port.address}.`);
                duplex.cancel();
            }
            finally {
                this.serialConnection = undefined;
                this.updateWsConfigParam({ connected: !!this.serialConnection });
                this.messages.length = 0;
                setTimeout(() => {
                    resolve(serial_service_1.Status.OK);
                }, 200);
            }
        });
    }
    async sendMessageToSerial(message) {
        if (!this.serialConnection) {
            return serial_service_1.Status.NOT_CONNECTED;
        }
        const req = new monitor_pb_1.StreamingOpenRequest();
        req.setData(new util_1.TextEncoder().encode(message));
        return new Promise((resolve) => {
            if (this.serialConnection) {
                this.serialConnection.duplex.write(req, () => {
                    resolve(serial_service_1.Status.OK);
                });
                return;
            }
            this.disconnect().then(() => resolve(serial_service_1.Status.NOT_CONNECTED));
        });
    }
    mapType(type) {
        switch (type) {
            case serial_service_1.SerialConfig.ConnectionType.SERIAL:
                return monitor_pb_1.MonitorConfig.TargetType.TARGET_TYPE_SERIAL;
            default:
                return monitor_pb_1.MonitorConfig.TargetType.TARGET_TYPE_SERIAL;
        }
    }
};
SerialServiceImpl = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(logger_1.ILogger)),
    __param(0, inversify_1.named(exports.SerialServiceName)),
    __param(1, inversify_1.inject(monitor_client_provider_1.MonitorClientProvider)),
    __param(2, inversify_1.inject(web_socket_service_1.WebSocketService)),
    __metadata("design:paramtypes", [Object, monitor_client_provider_1.MonitorClientProvider, Object])
], SerialServiceImpl);
exports.SerialServiceImpl = SerialServiceImpl;
// converts 'ab\nc\nd' => [ab\n,c\n,d]
function stringToArray(string, separator = '\n') {
    const retArray = [];
    let prevChar = separator;
    for (let i = 0; i < string.length; i++) {
        const currChar = string[i];
        if (prevChar === separator) {
            retArray.push(currChar);
        }
        else {
            const lastWord = retArray[retArray.length - 1];
            retArray[retArray.length - 1] = lastWord + currChar;
        }
        prevChar = currChar;
    }
    return retArray;
}
//# sourceMappingURL=serial-service-impl.js.map