"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.spawnCommand = exports.getExecPath = void 0;
const os = require("os");
const which = require("which");
const semver = require("semver");
const path_1 = require("path");
const child_process_1 = require("child_process");
async function getExecPath(commandName, onError = (error) => console.log(error), versionArg, inBinDir) {
    const execName = `${commandName}${os.platform() === 'win32' ? '.exe' : ''}`;
    const relativePath = ['..', '..', 'build'];
    if (inBinDir) {
        relativePath.push('bin');
    }
    const buildCommand = path_1.join(__dirname, ...relativePath, execName);
    if (!versionArg) {
        return buildCommand;
    }
    const versionRegexp = /\d+\.\d+\.\d+/;
    const buildVersion = await spawnCommand(`"${buildCommand}"`, [versionArg], onError);
    const buildShortVersion = (buildVersion.match(versionRegexp) || [])[0];
    const pathCommand = await new Promise((resolve) => which(execName, (error, path) => resolve(error ? undefined : path)));
    if (!pathCommand) {
        return buildCommand;
    }
    const pathVersion = await spawnCommand(`"${pathCommand}"`, [versionArg], onError);
    const pathShortVersion = (pathVersion.match(versionRegexp) || [])[0];
    if (semver.gt(pathShortVersion, buildShortVersion)) {
        return pathCommand;
    }
    return buildCommand;
}
exports.getExecPath = getExecPath;
function spawnCommand(command, args, onError = (error) => console.log(error)) {
    return new Promise((resolve, reject) => {
        const cp = child_process_1.spawn(command, args, { windowsHide: true, shell: true });
        const outBuffers = [];
        const errBuffers = [];
        cp.stdout.on('data', (b) => outBuffers.push(b));
        cp.stderr.on('data', (b) => errBuffers.push(b));
        cp.on('error', (error) => {
            onError(error);
            reject(error);
        });
        cp.on('exit', (code, signal) => {
            if (code === 0) {
                const result = Buffer.concat(outBuffers).toString('utf8').trim();
                resolve(result);
                return;
            }
            if (errBuffers.length > 0) {
                const message = Buffer.concat(errBuffers).toString('utf8').trim();
                const error = new Error(`Error executing ${command} ${args.join(' ')}: ${message}`);
                onError(error);
                reject(error);
                return;
            }
            if (signal) {
                const error = new Error(`Process exited with signal: ${signal}`);
                onError(error);
                reject(error);
                return;
            }
            if (code) {
                const error = new Error(`Process exited with exit code: ${code}`);
                onError(error);
                reject(error);
                return;
            }
        });
    });
}
exports.spawnCommand = spawnCommand;
//# sourceMappingURL=exec-util.js.map