"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DaemonLog = void 0;
const logger_1 = require("@theia/core/lib/common/logger");
var DaemonLog;
(function (DaemonLog) {
    let Url;
    (function (Url) {
        function is(arg) {
            return (!!arg &&
                typeof arg.Scheme === 'string' &&
                typeof arg.Host === 'string' &&
                typeof arg.Path === 'string');
        }
        Url.is = is;
        function toString(url) {
            const { Scheme, Host, Path } = url;
            return `${Scheme}://${Host}${Path}`;
        }
        Url.toString = toString;
    })(Url = DaemonLog.Url || (DaemonLog.Url = {}));
    let System;
    (function (System) {
        function toString(system) {
            return `OS: ${system.os}`;
        }
        System.toString = toString;
    })(System = DaemonLog.System || (DaemonLog.System = {}));
    let Tool;
    (function (Tool) {
        function is(arg) {
            return !!arg && typeof arg.version === 'string' && 'systems' in arg;
        }
        Tool.is = is;
        function toString(tool) {
            const { version, systems } = tool;
            return `Version: ${version}${!!systems
                ? ` Systems: [${tool.systems.map(System.toString).join(', ')}]`
                : ''}`;
        }
        Tool.toString = toString;
    })(Tool = DaemonLog.Tool || (DaemonLog.Tool = {}));
    function is(arg) {
        return (!!arg &&
            typeof arg.time === 'string' &&
            typeof arg.level === 'string' &&
            typeof arg.msg === 'string');
    }
    DaemonLog.is = is;
    function toLogLevel(log) {
        const { level } = log;
        switch (level) {
            case 'trace':
                return logger_1.LogLevel.TRACE;
            case 'debug':
                return logger_1.LogLevel.DEBUG;
            case 'info':
                return logger_1.LogLevel.INFO;
            case 'warning':
                return logger_1.LogLevel.WARN;
            case 'error':
                return logger_1.LogLevel.ERROR;
            default:
                return logger_1.LogLevel.INFO;
        }
    }
    DaemonLog.toLogLevel = toLogLevel;
    function log(logger, logMessages) {
        const parsed = parse(logMessages);
        for (const log of parsed) {
            const logLevel = toLogLevel(log);
            const message = toMessage(log, { omitLogLevel: true });
            logger.log(logLevel, message);
        }
    }
    DaemonLog.log = log;
    function parse(toLog) {
        const messages = toLog.trim().split('\n');
        const result = [];
        for (let i = 0; i < messages.length; i++) {
            try {
                const maybeDaemonLog = JSON.parse(messages[i]);
                if (DaemonLog.is(maybeDaemonLog)) {
                    result.push(maybeDaemonLog);
                    continue;
                }
            }
            catch (_a) {
                /* NOOP */
            }
            result.push({
                time: new Date().toString(),
                level: 'info',
                msg: messages[i],
            });
        }
        return result;
    }
    function toPrettyString(logMessages) {
        const parsed = parse(logMessages);
        return parsed.map((log) => toMessage(log)).join('\n') + '\n';
    }
    DaemonLog.toPrettyString = toPrettyString;
    function toMessage(log, options = { omitLogLevel: false }) {
        const details = Object.keys(log)
            .filter((key) => key !== 'msg' && key !== 'level' && key !== 'time')
            .map((key) => toDetails(log, key))
            .join(', ');
        const logLevel = options.omitLogLevel
            ? ''
            : `[${log.level.toUpperCase()}] `;
        return `${logLevel}${log.msg}${!!details ? ` [${details}]` : ''}`;
    }
    function toDetails(log, key) {
        let value = log[key];
        if (DaemonLog.Url.is(value)) {
            value = DaemonLog.Url.toString(value);
        }
        else if (DaemonLog.Tool.is(value)) {
            value = DaemonLog.Tool.toString(value);
        }
        else if (typeof value === 'object') {
            value = JSON.stringify(value).replace(/\"([^(\")"]+)\":/g, '$1:'); // Remove the quotes from the property keys.
        }
        return `${key.toLowerCase()}: ${value}`;
    }
})(DaemonLog = exports.DaemonLog || (exports.DaemonLog = {}));
//# sourceMappingURL=daemon-log.js.map