"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigServiceImpl = void 0;
const fs = require("fs");
const path = require("path");
const temp = require("temp");
const yaml = require("js-yaml");
const util_1 = require("util");
const grpc = require("@grpc/grpc-js");
const inversify_1 = require("inversify");
const uri_1 = require("@theia/core/lib/common/uri");
const logger_1 = require("@theia/core/lib/common/logger");
const file_uri_1 = require("@theia/core/lib/node/file-uri");
const event_1 = require("@theia/core/lib/common/event");
const protocol_1 = require("../common/protocol");
const exec_util_1 = require("./exec-util");
const settings_pb_1 = require("./cli-protocol/cc/arduino/cli/settings/v1/settings_pb");
const serviceGrpcPb = require("./cli-protocol/cc/arduino/cli/settings/v1/settings_grpc_pb");
const arduino_daemon_impl_1 = require("./arduino-daemon-impl");
const cli_config_1 = require("./cli-config");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const core_1 = require("@theia/core");
const deepmerge = require('deepmerge');
const track = temp.track();
let ConfigServiceImpl = class ConfigServiceImpl {
    constructor() {
        this.ready = new promise_util_1.Deferred();
        this.configChangeEmitter = new event_1.Emitter();
    }
    async onStart() {
        await this.ensureCliConfigExists();
        this.cliConfig = await this.loadCliConfig();
        if (this.cliConfig) {
            const config = await this.mapCliConfigToAppConfig(this.cliConfig);
            if (config) {
                this.config = config;
                this.ready.resolve();
                return;
            }
        }
        this.fireInvalidConfig();
    }
    async getCliConfigFileUri() {
        const configDirUri = await this.envVariablesServer.getConfigDirUri();
        return new uri_1.default(configDirUri).resolve(cli_config_1.CLI_CONFIG).toString();
    }
    async getConfiguration() {
        await this.ready.promise;
        await this.daemon.ready;
        return Object.assign(Object.assign({}, this.config), { daemon: { port: await this.daemon.getPort() } });
    }
    // Used by frontend to update the config.
    async setConfiguration(config) {
        await this.ready.promise;
        if (protocol_1.Config.sameAs(this.config, config)) {
            return;
        }
        let copyDefaultCliConfig = core_1.deepClone(this.cliConfig);
        if (!copyDefaultCliConfig) {
            copyDefaultCliConfig = await this.getFallbackCliConfig();
        }
        const { additionalUrls, dataDirUri, downloadsDirUri, sketchDirUri, network, locale, } = config;
        copyDefaultCliConfig.directories = {
            data: file_uri_1.FileUri.fsPath(dataDirUri),
            downloads: file_uri_1.FileUri.fsPath(downloadsDirUri),
            user: file_uri_1.FileUri.fsPath(sketchDirUri),
        };
        copyDefaultCliConfig.board_manager = {
            additional_urls: [...additionalUrls],
        };
        copyDefaultCliConfig.locale = locale || 'en';
        const proxy = protocol_1.Network.stringify(network);
        copyDefaultCliConfig.network = { proxy };
        // always use the port of the daemon
        const port = await this.daemon.getPort();
        await this.updateDaemon(port, copyDefaultCliConfig);
        await this.writeDaemonState(port);
        this.config = core_1.deepClone(config);
        this.cliConfig = copyDefaultCliConfig;
        this.fireConfigChanged(this.config);
    }
    get cliConfiguration() {
        return this.cliConfig;
    }
    get onConfigChange() {
        return this.configChangeEmitter.event;
    }
    async getVersion() {
        return this.daemon.getVersion();
    }
    async isInDataDir(uri) {
        return this.getConfiguration().then(({ dataDirUri }) => new uri_1.default(dataDirUri).isEqualOrParent(new uri_1.default(uri)));
    }
    async isInSketchDir(uri) {
        return this.getConfiguration().then(({ sketchDirUri }) => new uri_1.default(sketchDirUri).isEqualOrParent(new uri_1.default(uri)));
    }
    async loadCliConfig() {
        const cliConfigFileUri = await this.getCliConfigFileUri();
        const cliConfigPath = file_uri_1.FileUri.fsPath(cliConfigFileUri);
        try {
            const content = await util_1.promisify(fs.readFile)(cliConfigPath, {
                encoding: 'utf8',
            });
            const model = yaml.safeLoad(content) || {};
            // The CLI can run with partial (missing `port`, `directories`), the app cannot, we merge the default with the user's config.
            const fallbackModel = await this.getFallbackCliConfig();
            return deepmerge(fallbackModel, model);
        }
        catch (error) {
            this.logger.error(`Error occurred when loading CLI config from ${cliConfigPath}.`, error);
        }
        return undefined;
    }
    async getFallbackCliConfig() {
        const cliPath = await this.daemon.getExecPath();
        const throwawayDirPath = await new Promise((resolve, reject) => {
            track.mkdir({}, (err, dirPath) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(dirPath);
            });
        });
        await exec_util_1.spawnCommand(`"${cliPath}"`, [
            'config',
            'init',
            '--dest-dir',
            `"${throwawayDirPath}"`,
        ]);
        const rawYaml = await util_1.promisify(fs.readFile)(path.join(throwawayDirPath, cli_config_1.CLI_CONFIG), { encoding: 'utf-8' });
        const model = yaml.safeLoad(rawYaml.trim());
        return model;
    }
    async ensureCliConfigExists() {
        const cliConfigFileUri = await this.getCliConfigFileUri();
        const cliConfigPath = file_uri_1.FileUri.fsPath(cliConfigFileUri);
        let exists = await util_1.promisify(fs.exists)(cliConfigPath);
        if (!exists) {
            await this.initCliConfigTo(path.dirname(cliConfigPath));
            exists = await util_1.promisify(fs.exists)(cliConfigPath);
            if (!exists) {
                throw new Error(`Could not initialize the default CLI configuration file at ${cliConfigPath}.`);
            }
        }
    }
    async initCliConfigTo(fsPathToDir) {
        const cliPath = await this.daemon.getExecPath();
        await exec_util_1.spawnCommand(`"${cliPath}"`, [
            'config',
            'init',
            '--dest-dir',
            `"${fsPathToDir}"`,
        ]);
    }
    async mapCliConfigToAppConfig(cliConfig) {
        var _a;
        const { directories, locale = 'en', daemon } = cliConfig;
        const { data, user, downloads } = directories;
        const additionalUrls = [];
        if (cliConfig.board_manager && cliConfig.board_manager.additional_urls) {
            additionalUrls.push(...Array.from(new Set(cliConfig.board_manager.additional_urls)));
        }
        const network = protocol_1.Network.parse((_a = cliConfig.network) === null || _a === void 0 ? void 0 : _a.proxy);
        return {
            dataDirUri: file_uri_1.FileUri.create(data).toString(),
            sketchDirUri: file_uri_1.FileUri.create(user).toString(),
            downloadsDirUri: file_uri_1.FileUri.create(downloads).toString(),
            additionalUrls,
            network,
            locale,
            daemon,
        };
    }
    fireConfigChanged(config) {
        this.configChangeEmitter.fire(config);
        this.notificationService.notifyConfigChanged({ config });
    }
    fireInvalidConfig() {
        this.notificationService.notifyConfigChanged({ config: undefined });
    }
    async updateDaemon(port, config) {
        const client = this.createClient(port);
        const req = new settings_pb_1.MergeRequest();
        const json = JSON.stringify(config, null, 2);
        req.setJsonData(json);
        console.log(`Updating daemon with 'data': ${json}`);
        return new Promise((resolve, reject) => {
            client.merge(req, (error) => {
                try {
                    if (error) {
                        reject(error);
                        return;
                    }
                    resolve();
                }
                finally {
                    client.close();
                }
            });
        });
    }
    async writeDaemonState(port) {
        const client = this.createClient(port);
        const req = new settings_pb_1.WriteRequest();
        const cliConfigUri = await this.getCliConfigFileUri();
        const cliConfigPath = file_uri_1.FileUri.fsPath(cliConfigUri);
        req.setFilePath(cliConfigPath);
        return new Promise((resolve, reject) => {
            client.write(req, (error) => {
                try {
                    if (error) {
                        reject(error);
                        return;
                    }
                    resolve();
                }
                finally {
                    client.close();
                }
            });
        });
    }
    createClient(port) {
        // https://github.com/agreatfool/grpc_tools_node_protoc_ts/blob/master/doc/grpcjs_support.md#usage
        const SettingsServiceClient = grpc.makeClientConstructor(
        // @ts-expect-error: ignore
        serviceGrpcPb['cc.arduino.cli.settings.v1.SettingsService'], 'SettingsServiceService');
        return new SettingsServiceClient(`localhost:${port}`, grpc.credentials.createInsecure());
    }
};
__decorate([
    inversify_1.inject(logger_1.ILogger),
    inversify_1.named('config'),
    __metadata("design:type", Object)
], ConfigServiceImpl.prototype, "logger", void 0);
__decorate([
    inversify_1.inject(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], ConfigServiceImpl.prototype, "envVariablesServer", void 0);
__decorate([
    inversify_1.inject(arduino_daemon_impl_1.ArduinoDaemonImpl),
    __metadata("design:type", arduino_daemon_impl_1.ArduinoDaemonImpl)
], ConfigServiceImpl.prototype, "daemon", void 0);
__decorate([
    inversify_1.inject(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], ConfigServiceImpl.prototype, "notificationService", void 0);
ConfigServiceImpl = __decorate([
    inversify_1.injectable()
], ConfigServiceImpl);
exports.ConfigServiceImpl = ConfigServiceImpl;
//# sourceMappingURL=config-service-impl.js.map