"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardDiscovery = void 0;
const inversify_1 = require("inversify");
const logger_1 = require("@theia/core/lib/common/logger");
const objects_1 = require("@theia/core/lib/common/objects");
const core_client_provider_1 = require("./core-client-provider");
const board_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/board_pb");
const protocol_1 = require("../common/protocol");
/**
 * Singleton service for tracking the available ports and board and broadcasting the
 * changes to all connected frontend instances. \
 * Unlike other services, this is not connection scoped.
 */
let BoardDiscovery = class BoardDiscovery extends core_client_provider_1.CoreClientAware {
    constructor() {
        super(...arguments);
        /**
         * Keys are the `address` of the ports. \
         * The `protocol` is ignored because the board detach event does not carry the protocol information,
         * just the address.
         * ```json
         * {
         *  "type": "remove",
         *  "address": "/dev/cu.usbmodem14101"
         * }
         * ```
         */
        this._state = {};
    }
    get state() {
        return this._state;
    }
    async init() {
        await this.coreClientProvider.initialized;
        const coreClient = await this.coreClient();
        this.startBoardListWatch(coreClient);
    }
    stopBoardListWatch(coreClient) {
        return new Promise((resolve, reject) => {
            if (!this.boardWatchDuplex) {
                return resolve();
            }
            const { instance } = coreClient;
            const req = new board_pb_1.BoardListWatchRequest();
            req.setInstance(instance);
            try {
                this.boardWatchDuplex.write(req.setInterrupt(true), resolve);
            }
            catch (e) {
                this.discoveryLogger.error(e);
                resolve();
            }
        });
    }
    startBoardListWatch(coreClient) {
        if (this.watching) {
            // We want to avoid starting the board list watch process multiple
            // times to meet unforseen consequences
            return;
        }
        this.watching = true;
        const { client, instance } = coreClient;
        const req = new board_pb_1.BoardListWatchRequest();
        req.setInstance(instance);
        this.boardWatchDuplex = client.boardListWatch();
        this.boardWatchDuplex.on('end', () => {
            this.watching = false;
            console.info('board watch ended');
        });
        this.boardWatchDuplex.on('close', () => {
            this.watching = false;
            console.info('board watch ended');
        });
        this.boardWatchDuplex.on('data', (resp) => {
            if (resp.getEventType() === 'quit') {
                this.watching = false;
                console.info('board watch ended');
                return;
            }
            const detectedPort = resp.getPort();
            if (detectedPort) {
                let eventType = 'unknown';
                if (resp.getEventType() === 'add') {
                    eventType = 'add';
                }
                else if (resp.getEventType() === 'remove') {
                    eventType = 'remove';
                }
                else {
                    eventType = 'unknown';
                }
                if (eventType === 'unknown') {
                    throw new Error(`Unexpected event type: '${resp.getEventType()}'`);
                }
                const oldState = objects_1.deepClone(this._state);
                const newState = objects_1.deepClone(this._state);
                const address = detectedPort.getPort().getAddress();
                const protocol = detectedPort.getPort().getProtocol();
                // Different discoveries can detect the same port with different
                // protocols, so we consider the combination of address and protocol
                // to be the id of a certain port to distinguish it from others.
                // If we'd use only the address of a port to store it in a map
                // we can have conflicts the same port is found with multiple
                // protocols.
                const portID = `${address}|${protocol}`;
                const label = detectedPort.getPort().getLabel();
                const protocolLabel = detectedPort.getPort().getProtocolLabel();
                const port = {
                    id: portID,
                    address,
                    addressLabel: label,
                    protocol,
                    protocolLabel,
                };
                const boards = [];
                for (const item of detectedPort.getMatchingBoardsList()) {
                    boards.push({
                        fqbn: item.getFqbn(),
                        name: item.getName() || 'unknown',
                        port,
                    });
                }
                if (eventType === 'add') {
                    if (newState[portID]) {
                        const [, knownBoards] = newState[portID];
                        console.warn(`Port '${protocol_1.Port.toString(port)}' was already available. Known boards before override: ${JSON.stringify(knownBoards)}`);
                    }
                    newState[portID] = [port, boards];
                }
                else if (eventType === 'remove') {
                    if (!newState[portID]) {
                        console.warn(`Port '${protocol_1.Port.toString(port)}' was not available. Skipping`);
                        return;
                    }
                    delete newState[portID];
                }
                const oldAvailablePorts = this.getAvailablePorts(oldState);
                const oldAttachedBoards = this.getAttachedBoards(oldState);
                const newAvailablePorts = this.getAvailablePorts(newState);
                const newAttachedBoards = this.getAttachedBoards(newState);
                const event = {
                    oldState: {
                        ports: oldAvailablePorts,
                        boards: oldAttachedBoards,
                    },
                    newState: {
                        ports: newAvailablePorts,
                        boards: newAttachedBoards,
                    },
                };
                this._state = newState;
                this.notificationService.notifyAttachedBoardsChanged(event);
            }
        });
        this.boardWatchDuplex.write(req);
    }
    getAttachedBoards(state = this.state) {
        const attachedBoards = [];
        for (const portID of Object.keys(state)) {
            const [, boards] = state[portID];
            attachedBoards.push(...boards);
        }
        return attachedBoards;
    }
    getAvailablePorts(state = this.state) {
        const availablePorts = [];
        for (const portID of Object.keys(state)) {
            const [port] = state[portID];
            availablePorts.push(port);
        }
        return availablePorts;
    }
};
__decorate([
    inversify_1.inject(logger_1.ILogger),
    inversify_1.named('discovery'),
    __metadata("design:type", Object)
], BoardDiscovery.prototype, "discoveryLogger", void 0);
__decorate([
    inversify_1.inject(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], BoardDiscovery.prototype, "notificationService", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], BoardDiscovery.prototype, "init", null);
BoardDiscovery = __decorate([
    inversify_1.injectable()
], BoardDiscovery);
exports.BoardDiscovery = BoardDiscovery;
//# sourceMappingURL=board-discovery.js.map