"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArduinoAuthenticationProvider = void 0;
const node_fetch_1 = require("node-fetch");
const inversify_1 = require("inversify");
const authentication_server_1 = require("./authentication-server");
const keychain_1 = require("./keychain");
const utils_1 = require("./utils");
const auth0_js_1 = require("auth0-js");
const event_1 = require("@theia/core/lib/common/event");
const open = require("open");
const LOGIN_TIMEOUT = 30 * 1000;
const REFRESH_INTERVAL = 10 * 60 * 1000;
let ArduinoAuthenticationProvider = class ArduinoAuthenticationProvider {
    constructor() {
        this.id = 'arduino-account-auth';
        this.label = 'Arduino';
        // create a keychain holding the keys
        this.keychain = new keychain_1.Keychain({
            credentialsSection: this.id,
            account: this.label,
        });
        this._tokens = [];
        this._refreshTimeouts = new Map();
        this._onDidChangeSessions = new event_1.Emitter();
    }
    get onDidChangeSessions() {
        return this._onDidChangeSessions.event;
    }
    get sessions() {
        return Promise.resolve(this._tokens.map((token) => utils_1.IToken2Session(token)));
    }
    getSessions() {
        return Promise.resolve(this.sessions);
    }
    async init() {
        // restore previously stored sessions
        const stringTokens = await this.keychain.getStoredCredentials();
        // no valid token, nothing to do
        if (!stringTokens) {
            return;
        }
        const checkToken = async () => {
            // tokens exist, parse and refresh them
            try {
                const tokens = JSON.parse(stringTokens);
                // refresh the tokens when needed
                await Promise.all(tokens.map(async (token) => {
                    // if refresh not needed, add the existing token
                    if (!utils_1.IToken.requiresRefresh(token, REFRESH_INTERVAL)) {
                        return this.addToken(token);
                    }
                    const refreshedToken = await this.refreshToken(token);
                    return this.addToken(refreshedToken);
                }));
            }
            catch (_a) {
                return;
            }
        };
        checkToken();
        setInterval(checkToken, REFRESH_INTERVAL);
    }
    setOptions(authOptions) {
        this.authOptions = authOptions;
    }
    dispose() { }
    async refreshToken(token) {
        if (!token.refreshToken) {
            throw new Error('Unable to refresh a token without a refreshToken');
        }
        console.log(`Refreshing token ${token.sessionId}`);
        const response = await node_fetch_1.default(`https://${this.authOptions.domain}/oauth/token`, {
            method: 'POST',
            body: JSON.stringify({
                grant_type: 'refresh_token',
                client_id: this.authOptions.clientID,
                refresh_token: token.refreshToken,
            }),
            headers: {
                'Content-Type': 'application/json',
                Accept: 'application/json',
            },
        });
        if (response.ok) {
            const result = await response.json();
            // add the refresh_token from the old token
            return utils_1.token2IToken(Object.assign(Object.assign({}, result), { refresh_token: token.refreshToken }));
        }
        throw new Error(`Failed to refresh a token: ${response.statusText}`);
    }
    async exchangeCodeForToken(authCode, verifier) {
        const response = await node_fetch_1.default(`https://${this.authOptions.domain}/oauth/token`, {
            method: 'POST',
            body: JSON.stringify({
                grant_type: 'authorization_code',
                client_id: this.authOptions.clientID,
                code_verifier: verifier,
                code: authCode,
                redirect_uri: this.authOptions.redirectUri,
            }),
            headers: {
                'Content-Type': 'application/json',
                Accept: 'application/json',
            },
        });
        if (response.ok) {
            return await response.json();
        }
        throw new Error(`Failed to fetch a token: ${response.statusText}`);
    }
    async createSession() {
        const token = await this.login();
        this.addToken(token);
        return utils_1.IToken2Session(token);
    }
    async login() {
        return new Promise(async (resolve, reject) => {
            const pkp = utils_1.generateProofKeyPair();
            const server = authentication_server_1.createServer(async (req, res) => {
                const { url } = req;
                if (url && url.startsWith('/callback?code=')) {
                    const code = url.slice('/callback?code='.length);
                    const token = await this.exchangeCodeForToken(code, pkp.verifier);
                    resolve(utils_1.token2IToken(token));
                }
                // schedule server shutdown after 10 seconds
                setTimeout(() => {
                    server.close();
                }, LOGIN_TIMEOUT);
            });
            try {
                const port = await authentication_server_1.startServer(server);
                console.log(`server listening on http://localhost:${port}`);
                const auth0 = new auth0_js_1.Authentication({
                    clientID: this.authOptions.clientID,
                    domain: this.authOptions.domain,
                    audience: this.authOptions.audience,
                    redirectUri: `http://localhost:${port}/callback`,
                    scope: this.authOptions.scopes.join(' '),
                    responseType: this.authOptions.responseType,
                    code_challenge_method: 'S256',
                    code_challenge: pkp.challenge,
                });
                const authorizeUrl = auth0.buildAuthorizeUrl({
                    redirectUri: `http://localhost:${port}/callback`,
                    responseType: this.authOptions.responseType,
                });
                await open(authorizeUrl);
                // set a timeout if the authentication takes too long
                setTimeout(() => {
                    server.close();
                    reject(new Error('Login timeout.'));
                }, 30000);
            }
            finally {
                // server is usually closed by the callback or the timeout, this is to handle corner cases
                setTimeout(() => {
                    server.close();
                }, 50000);
            }
        });
    }
    async signUp() {
        await open(this.authOptions.registerUri);
    }
    /**
     * Returns extended account info for the given (and logged-in) sessionId.
     *
     * @param sessionId the sessionId to get info about. If not provided, all account info are returned
     * @returns an array of IToken, containing extended info for the accounts
     */
    accountInfo(sessionId) {
        return this._tokens.filter((token) => sessionId ? token.sessionId === sessionId : true);
    }
    /**
     * Removes any logged-in sessions
     */
    logout() {
        this._tokens.forEach((token) => this.removeSession(token.sessionId));
        // remove any dangling credential in the keychain
        this.keychain.deleteCredentials();
    }
    async removeSession(sessionId) {
        // remove token from memory, if successful fire the event
        const token = this.removeInMemoryToken(sessionId);
        if (token) {
            this._onDidChangeSessions.fire({
                added: [],
                removed: [utils_1.IToken2Session(token)],
                changed: [],
            });
        }
        // update the tokens in the keychain
        this.keychain.storeCredentials(JSON.stringify(this._tokens));
    }
    /**
     * Clears the refresh timeout associated to a session and removes the key from the set
     */
    clearSessionTimeout(sessionId) {
        const timeout = this._refreshTimeouts.get(sessionId);
        if (timeout) {
            clearTimeout(timeout);
            this._refreshTimeouts.delete(sessionId);
        }
    }
    /**
     * Remove the given token from memory and clears the associated refresh timeout
     * @param token
     * @returns the removed token
     */
    removeInMemoryToken(sessionId) {
        const tokenIndex = this._tokens.findIndex((token) => token.sessionId === sessionId);
        let token;
        if (tokenIndex > -1) {
            token = this._tokens[tokenIndex];
            this._tokens.splice(tokenIndex, 1);
        }
        this.clearSessionTimeout(sessionId);
        return token;
    }
    /**
     * Add the given token to memory storage and keychain. Prepares Timeout for token refresh
     * NOTE: we currently support 1 token (logged user) at a time
     * @param token
     * @returns
     */
    async addToken(token) {
        if (!token) {
            return;
        }
        this._tokens = [token];
        // update the tokens in the keychain
        this.keychain.storeCredentials(JSON.stringify(this._tokens));
        // notify subscribers about the newly added/changed session
        const session = utils_1.IToken2Session(token);
        const changedToken = this._tokens.find((itoken) => itoken.sessionId === session.id);
        const changes = {
            added: (!changedToken && [session]) || [],
            removed: [],
            changed: (!!changedToken && [session]) || [],
        };
        this._onDidChangeSessions.fire(changes);
        // setup token refresh
        this.clearSessionTimeout(token.sessionId);
        if (token.expiresAt) {
            // refresh the token 30sec before expiration
            const expiration = token.expiresAt - Date.now() - 30 * 1000;
            this._refreshTimeouts.set(token.sessionId, setTimeout(async () => {
                try {
                    const refreshedToken = await this.refreshToken(token);
                    this.addToken(refreshedToken);
                }
                catch (e) {
                    await this.removeSession(token.sessionId);
                }
            }, expiration > 0 ? expiration : 0));
        }
        return token;
    }
};
ArduinoAuthenticationProvider = __decorate([
    inversify_1.injectable()
], ArduinoAuthenticationProvider);
exports.ArduinoAuthenticationProvider = ArduinoAuthenticationProvider;
//# sourceMappingURL=arduino-auth-provider.js.map