"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DaemonError = exports.ArduinoDaemonImpl = void 0;
const path_1 = require("path");
const inversify_1 = require("inversify");
const child_process_1 = require("child_process");
const file_uri_1 = require("@theia/core/lib/node/file-uri");
const logger_1 = require("@theia/core/lib/common/logger");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const disposable_1 = require("@theia/core/lib/common/disposable");
const event_1 = require("@theia/core/lib/common/event");
const environment_1 = require("@theia/application-package/lib/environment");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const localization_provider_1 = require("@theia/core/lib/node/i18n/localization-provider");
const protocol_1 = require("../common/protocol");
const daemon_log_1 = require("./daemon-log");
const cli_config_1 = require("./cli-config");
const exec_util_1 = require("./exec-util");
let ArduinoDaemonImpl = class ArduinoDaemonImpl {
    constructor() {
        this.toDispose = new disposable_1.DisposableCollection();
        this.onDaemonStartedEmitter = new event_1.Emitter();
        this.onDaemonStoppedEmitter = new event_1.Emitter();
        this._running = false;
        this._ready = new promise_util_1.Deferred();
    }
    // Backend application lifecycle.
    onStart() {
        this.startDaemon();
    }
    // Daemon API
    async isRunning() {
        return Promise.resolve(this._running);
    }
    async getPort() {
        return Promise.resolve(this._port);
    }
    async startDaemon() {
        try {
            this.toDispose.dispose(); // This will `kill` the previously started daemon process, if any.
            const cliPath = await this.getExecPath();
            this.onData(`Starting daemon from ${cliPath}...`);
            const { daemon, port } = await this.spawnDaemonProcess();
            this._port = port;
            // Watchdog process for terminating the daemon process when the backend app terminates.
            child_process_1.spawn(process.execPath, [
                path_1.join(__dirname, 'daemon-watcher.js'),
                String(process.pid),
                String(daemon.pid),
            ], {
                env: environment_1.environment.electron.runAsNodeEnv(),
                detached: true,
                stdio: 'ignore',
                windowsHide: true,
            }).unref();
            this.toDispose.pushAll([
                disposable_1.Disposable.create(() => daemon.kill()),
                disposable_1.Disposable.create(() => this.fireDaemonStopped()),
            ]);
            this.fireDaemonStarted();
            this.onData('Daemon is running.');
        }
        catch (err) {
            this.onData('Failed to start the daemon.');
            this.onError(err);
            let i = 5; // TODO: make this better
            while (i) {
                this.onData(`Restarting daemon in ${i} seconds...`);
                await new Promise((resolve) => setTimeout(resolve, 1000));
                i--;
            }
            this.onData('Restarting daemon now...');
            return this.startDaemon();
        }
    }
    async stopDaemon() {
        this.toDispose.dispose();
    }
    get onDaemonStarted() {
        return this.onDaemonStartedEmitter.event;
    }
    get onDaemonStopped() {
        return this.onDaemonStoppedEmitter.event;
    }
    get ready() {
        return this._ready.promise;
    }
    async getExecPath() {
        if (this._execPath) {
            return this._execPath;
        }
        this._execPath = await exec_util_1.getExecPath('arduino-cli', this.onError.bind(this));
        return this._execPath;
    }
    async getVersion() {
        const execPath = await this.getExecPath();
        const raw = await exec_util_1.spawnCommand(`"${execPath}"`, ['version', '--format', 'json'], this.onError.bind(this));
        try {
            // The CLI `Info` object: https://github.com/arduino/arduino-cli/blob/17d24eb901b1fdaa5a4ec7da3417e9e460f84007/version/version.go#L31-L34
            const { VersionString, Commit, Status } = JSON.parse(raw);
            return {
                version: VersionString,
                commit: Commit,
                status: Status,
            };
        }
        catch (_a) {
            return { version: raw, commit: raw };
        }
    }
    async getSpawnArgs() {
        const configDirUri = await this.envVariablesServer.getConfigDirUri();
        const cliConfigPath = path_1.join(file_uri_1.FileUri.fsPath(configDirUri), cli_config_1.CLI_CONFIG);
        return [
            'daemon',
            '--format',
            'jsonmini',
            '--port',
            '0',
            '--config-file',
            `"${cliConfigPath}"`,
            '-v',
            '--log-format',
            'json',
        ];
    }
    async spawnDaemonProcess() {
        const [cliPath, args] = await Promise.all([
            this.getExecPath(),
            this.getSpawnArgs(),
        ]);
        const ready = new promise_util_1.Deferred();
        const options = { shell: true };
        const daemon = child_process_1.spawn(`"${cliPath}"`, args, options);
        // If the process exists right after the daemon gRPC server has started (due to an invalid port, unknown address, TCP port in use, etc.)
        // we have no idea about the root cause unless we sniff into the first data package and dispatch the logic on that. Note, we get a exit code 1.
        let grpcServerIsReady = false;
        daemon.stdout.on('data', (data) => {
            const message = data.toString();
            let port = '';
            let address = '';
            message
                .split('\n')
                .filter((line) => line.length)
                .forEach((line) => {
                try {
                    const parsedLine = JSON.parse(line);
                    if ('Port' in parsedLine) {
                        port = parsedLine.Port;
                    }
                    if ('IP' in parsedLine) {
                        address = parsedLine.IP;
                    }
                }
                catch (err) {
                    // ignore
                }
            });
            this.onData(message);
            if (!grpcServerIsReady) {
                const error = DaemonError.parse(message);
                if (error) {
                    ready.reject(error);
                    return;
                }
                if (port.length && address.length) {
                    grpcServerIsReady = true;
                    ready.resolve({ daemon, port });
                }
            }
        });
        daemon.stderr.on('data', (data) => {
            const message = data.toString();
            this.onData(data.toString());
            const error = DaemonError.parse(message);
            ready.reject(error ? error : new Error(data.toString().trim()));
        });
        daemon.on('exit', (code, signal) => {
            if (code === 0 || signal === 'SIGINT' || signal === 'SIGKILL') {
                this.onData('Daemon has stopped.');
            }
            else {
                this.onData(`Daemon exited with ${typeof code === 'undefined'
                    ? `signal '${signal}'`
                    : `exit code: ${code}`}.`);
            }
        });
        daemon.on('error', (error) => {
            this.onError(error);
            ready.reject(error);
        });
        return ready.promise;
    }
    fireDaemonStarted() {
        this._running = true;
        this._ready.resolve();
        this.onDaemonStartedEmitter.fire();
        this.notificationService.notifyDaemonStarted();
    }
    fireDaemonStopped() {
        if (!this._running) {
            return;
        }
        this._running = false;
        this._ready.reject(); // Reject all pending.
        this._ready = new promise_util_1.Deferred();
        this.onDaemonStoppedEmitter.fire();
        this.notificationService.notifyDaemonStopped();
    }
    onData(message) {
        daemon_log_1.DaemonLog.log(this.logger, message);
    }
    onError(error) {
        this.logger.error(error);
    }
};
__decorate([
    inversify_1.inject(logger_1.ILogger),
    inversify_1.named('daemon'),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "logger", void 0);
__decorate([
    inversify_1.inject(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "envVariablesServer", void 0);
__decorate([
    inversify_1.inject(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "notificationService", void 0);
__decorate([
    inversify_1.inject(localization_provider_1.LocalizationProvider),
    __metadata("design:type", localization_provider_1.LocalizationProvider)
], ArduinoDaemonImpl.prototype, "localizationProvider", void 0);
ArduinoDaemonImpl = __decorate([
    inversify_1.injectable()
], ArduinoDaemonImpl);
exports.ArduinoDaemonImpl = ArduinoDaemonImpl;
class DaemonError extends Error {
    constructor(message, code, details) {
        super(message);
        this.code = code;
        this.details = details;
        Object.setPrototypeOf(this, DaemonError.prototype);
    }
}
exports.DaemonError = DaemonError;
(function (DaemonError) {
    DaemonError.ADDRESS_IN_USE = 0;
    DaemonError.UNKNOWN_ADDRESS = 2;
    DaemonError.INVALID_PORT = 4;
    DaemonError.UNKNOWN = 8;
    function parse(log) {
        const raw = log.toLocaleLowerCase();
        if (raw.includes('failed to listen')) {
            if (raw.includes('address already in use') ||
                (raw.includes('bind') &&
                    raw.includes('only one usage of each socket address'))) {
                return new DaemonError('Failed to listen on TCP port. Address already in use.', DaemonError.ADDRESS_IN_USE);
            }
            if (raw.includes('is unknown name') ||
                (raw.includes('tcp/') && raw.includes('is an invalid port'))) {
                return new DaemonError('Failed to listen on TCP port. Unknown address.', DaemonError.UNKNOWN_ADDRESS);
            }
            if (raw.includes('is an invalid port')) {
                return new DaemonError('Failed to listen on TCP port. Invalid port.', DaemonError.INVALID_PORT);
            }
        }
        // Based on the CLI logging: `failed to serve`, and  any other FATAL errors.
        // https://github.com/arduino/arduino-cli/blob/11abbee8a9f027d087d4230f266a87217677d423/cli/daemon/daemon.go#L89-L94
        if (raw.includes('failed to serve') &&
            (raw.includes('"fatal"') || raw.includes('fata'))) {
            return new DaemonError('Unexpected CLI start error.', DaemonError.UNKNOWN, log);
        }
        return undefined;
    }
    DaemonError.parse = parse;
})(DaemonError = exports.DaemonError || (exports.DaemonError = {}));
//# sourceMappingURL=arduino-daemon-impl.js.map