"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Installable = void 0;
const semver = require("semver");
const cancellation_1 = require("@theia/core/lib/common/cancellation");
const utils_1 = require("./../utils");
var Installable;
(function (Installable) {
    let Version;
    (function (Version) {
        /**
         * Most recent version comes first, then the previous versions. (`1.8.1`, `1.6.3`, `1.6.2`, `1.6.1` and so on.)
         */
        Version.COMPARATOR = (left, right) => {
            if (semver.valid(left) && semver.valid(right)) {
                return semver.compare(left, right);
            }
            return utils_1.naturalCompare(left, right);
        };
    })(Version = Installable.Version || (Installable.Version = {}));
    async function installWithProgress(options) {
        const { item, version } = options;
        return doWithProgress(Object.assign(Object.assign({}, options), { progressText: `Processing ${item.name}:${version}`, run: ({ progressId }) => options.installable.install({
                item: options.item,
                version: options.version,
                progressId,
            }) }));
    }
    Installable.installWithProgress = installWithProgress;
    async function uninstallWithProgress(options) {
        const { item } = options;
        return doWithProgress(Object.assign(Object.assign({}, options), { progressText: `Processing ${item.name}${item.installedVersion ? `:${item.installedVersion}` : ''}`, run: ({ progressId }) => options.installable.uninstall({
                item: options.item,
                progressId,
            }) }));
    }
    Installable.uninstallWithProgress = uninstallWithProgress;
    async function doWithProgress(options) {
        return withProgress(options.progressText, options.messageService, async (progress, _) => {
            const progressId = progress.id;
            const toDispose = options.responseService.onProgressDidChange((progressMessage) => {
                if (progressId === progressMessage.progressId) {
                    const { message, work } = progressMessage;
                    progress.report({ message, work });
                }
            });
            try {
                options.responseService.clearArduinoChannel();
                await options.run({ progressId });
            }
            finally {
                toDispose.dispose();
            }
        });
    }
    Installable.doWithProgress = doWithProgress;
    async function withProgress(text, messageService, cb) {
        const cancellationSource = new cancellation_1.CancellationTokenSource();
        const { token } = cancellationSource;
        const progress = await messageService.showProgress({ text, options: { cancelable: false } }, () => cancellationSource.cancel());
        try {
            await cb(progress, token);
        }
        finally {
            progress.cancel();
        }
    }
})(Installable = exports.Installable || (exports.Installable = {}));
//# sourceMappingURL=installable.js.map