"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Board = exports.Programmer = exports.ConfigOption = exports.InstalledBoardWithPackage = exports.BoardWithPackage = exports.BoardsPackage = exports.Port = exports.BoardsService = exports.BoardsServicePath = exports.AttachedBoardsChangeEvent = exports.AvailablePorts = void 0;
const utils_1 = require("./../utils");
var AvailablePorts;
(function (AvailablePorts) {
    function byProtocol(availablePorts) {
        const grouped = new Map();
        for (const portID of Object.keys(availablePorts)) {
            const [port, boards] = availablePorts[portID];
            let ports = grouped.get(port.protocol);
            if (!ports) {
                ports = {};
            }
            ports[portID] = [port, boards];
            grouped.set(port.protocol, ports);
        }
        return grouped;
    }
    AvailablePorts.byProtocol = byProtocol;
})(AvailablePorts = exports.AvailablePorts || (exports.AvailablePorts = {}));
var AttachedBoardsChangeEvent;
(function (AttachedBoardsChangeEvent) {
    function isEmpty(event) {
        const { detached, attached } = diff(event);
        return (!!detached.boards.length &&
            !!detached.ports.length &&
            !!attached.boards.length &&
            !!attached.ports.length);
    }
    AttachedBoardsChangeEvent.isEmpty = isEmpty;
    function toString(event) {
        const rows = [];
        if (!isEmpty(event)) {
            const { attached, detached } = diff(event);
            const visitedAttachedPorts = [];
            const visitedDetachedPorts = [];
            for (const board of attached.boards) {
                const port = board.port
                    ? ` on ${Port.toString(board.port)}`
                    : '';
                rows.push(` - Attached board: ${Board.toString(board)}${port}`);
                if (board.port) {
                    visitedAttachedPorts.push(board.port);
                }
            }
            for (const board of detached.boards) {
                const port = board.port
                    ? ` from ${Port.toString(board.port)}`
                    : '';
                rows.push(` - Detached board: ${Board.toString(board)}${port}`);
                if (board.port) {
                    visitedDetachedPorts.push(board.port);
                }
            }
            for (const port of attached.ports) {
                if (!visitedAttachedPorts.find((p) => Port.sameAs(port, p))) {
                    rows.push(` - New port is available on ${Port.toString(port)}`);
                }
            }
            for (const port of detached.ports) {
                if (!visitedDetachedPorts.find((p) => Port.sameAs(port, p))) {
                    rows.push(` - Port is no longer available on ${Port.toString(port)}`);
                }
            }
        }
        return rows.length ? rows.join('\n') : 'No changes.';
    }
    AttachedBoardsChangeEvent.toString = toString;
    function diff(event) {
        // In `lefts` AND not in `rights`.
        const diff = (lefts, rights, sameAs) => {
            return lefts.filter((left) => rights.findIndex((right) => sameAs(left, right)) === -1);
        };
        const { boards: newBoards } = event.newState;
        const { boards: oldBoards } = event.oldState;
        const { ports: newPorts } = event.newState;
        const { ports: oldPorts } = event.oldState;
        const boardSameAs = (left, right) => Board.sameAs(left, right);
        const portSameAs = (left, right) => Port.sameAs(left, right);
        return {
            detached: {
                boards: diff(oldBoards, newBoards, boardSameAs),
                ports: diff(oldPorts, newPorts, portSameAs),
            },
            attached: {
                boards: diff(newBoards, oldBoards, boardSameAs),
                ports: diff(newPorts, oldPorts, portSameAs),
            },
        };
    }
    AttachedBoardsChangeEvent.diff = diff;
})(AttachedBoardsChangeEvent = exports.AttachedBoardsChangeEvent || (exports.AttachedBoardsChangeEvent = {}));
exports.BoardsServicePath = '/services/boards-service';
exports.BoardsService = Symbol('BoardsService');
var Port;
(function (Port) {
    function is(arg) {
        return (!!arg &&
            'address' in arg &&
            typeof arg['address'] === 'string' &&
            'protocol' in arg &&
            typeof arg['protocol'] === 'string');
    }
    Port.is = is;
    function toString(port) {
        return `${port.addressLabel} ${port.protocolLabel}`;
    }
    Port.toString = toString;
    function compare(left, right) {
        // Ports must be sorted in this order:
        // 1. Serial
        // 2. Network
        // 3. Other protocols
        if (left.protocol === "serial" && right.protocol !== "serial") {
            return -1;
        }
        else if (left.protocol !== "serial" && right.protocol === "serial") {
            return 1;
        }
        else if (left.protocol === "network" && right.protocol !== "network") {
            return -1;
        }
        else if (left.protocol !== "network" && right.protocol === "network") {
            return 1;
        }
        return utils_1.naturalCompare(left.address, right.address);
    }
    Port.compare = compare;
    function sameAs(left, right) {
        if (left && right) {
            return left.address === right.address && left.protocol === right.protocol;
        }
        return false;
    }
    Port.sameAs = sameAs;
})(Port = exports.Port || (exports.Port = {}));
var BoardsPackage;
(function (BoardsPackage) {
    function equals(left, right) {
        return left.id === right.id;
    }
    BoardsPackage.equals = equals;
    function contains(selectedBoard, { id, boards }) {
        if (boards.some((board) => Board.sameAs(board, selectedBoard))) {
            return true;
        }
        if (selectedBoard.fqbn) {
            const [platform, architecture] = selectedBoard.fqbn.split(':');
            if (platform && architecture) {
                return `${platform}:${architecture}` === id;
            }
        }
        return false;
    }
    BoardsPackage.contains = contains;
})(BoardsPackage = exports.BoardsPackage || (exports.BoardsPackage = {}));
var BoardWithPackage;
(function (BoardWithPackage) {
    function is(board) {
        return !!board.packageId && !!board.packageName;
    }
    BoardWithPackage.is = is;
})(BoardWithPackage = exports.BoardWithPackage || (exports.BoardWithPackage = {}));
var InstalledBoardWithPackage;
(function (InstalledBoardWithPackage) {
    function is(boardWithPackage) {
        return !!boardWithPackage.fqbn;
    }
    InstalledBoardWithPackage.is = is;
})(InstalledBoardWithPackage = exports.InstalledBoardWithPackage || (exports.InstalledBoardWithPackage = {}));
var ConfigOption;
(function (ConfigOption) {
    function is(arg) {
        return (!!arg &&
            'option' in arg &&
            'label' in arg &&
            'values' in arg &&
            typeof arg['option'] === 'string' &&
            typeof arg['label'] === 'string' &&
            Array.isArray(arg['values']));
    }
    ConfigOption.is = is;
    /**
     * Appends the configuration options to the `fqbn` argument.
     * Throws an error if the `fqbn` does not have the `segment(':'segment)*` format.
     * The provided output format is always segment(':'segment)*(':'option'='value(','option'='value)*)?
     */
    function decorate(fqbn, configOptions) {
        if (!configOptions.length) {
            return fqbn;
        }
        const toValue = (values) => {
            const selectedValue = values.find(({ selected }) => selected);
            if (!selectedValue) {
                console.warn(`None of the config values was selected. Values were: ${JSON.stringify(values)}`);
                return undefined;
            }
            return selectedValue.value;
        };
        const options = configOptions
            .map(({ option, values }) => [option, toValue(values)])
            .filter(([, value]) => !!value)
            .map(([option, value]) => `${option}=${value}`)
            .join(',');
        return `${fqbn}:${options}`;
    }
    ConfigOption.decorate = decorate;
    class ConfigOptionError extends Error {
        constructor(message) {
            super(message);
            Object.setPrototypeOf(this, ConfigOptionError.prototype);
        }
    }
    ConfigOption.ConfigOptionError = ConfigOptionError;
    ConfigOption.LABEL_COMPARATOR = (left, right) => utils_1.naturalCompare(left.label.toLocaleLowerCase(), right.label.toLocaleLowerCase());
})(ConfigOption = exports.ConfigOption || (exports.ConfigOption = {}));
var Programmer;
(function (Programmer) {
    function equals(left, right) {
        if (!left) {
            return !right;
        }
        if (!right) {
            return !left;
        }
        return (left.id === right.id &&
            left.name === right.name &&
            left.platform === right.platform);
    }
    Programmer.equals = equals;
})(Programmer = exports.Programmer || (exports.Programmer = {}));
var Board;
(function (Board) {
    function is(board) {
        return !!board && 'name' in board;
    }
    Board.is = is;
    function equals(left, right) {
        return left.name === right.name && left.fqbn === right.fqbn;
    }
    Board.equals = equals;
    function sameAs(left, right) {
        // How to associate a selected board with one of the available cores: https://typefox.slack.com/archives/CJJHJCJSJ/p1571142327059200
        // 1. How to use the FQBN if any and infer the package ID from it: https://typefox.slack.com/archives/CJJHJCJSJ/p1571147549069100
        // 2. How to trim the `/Genuino` from the name: https://arduino.slack.com/archives/CJJHJCJSJ/p1571146951066800?thread_ts=1571142327.059200&cid=CJJHJCJSJ
        const other = typeof right === 'string' ? { name: right } : right;
        if (left.fqbn && other.fqbn) {
            return left.fqbn === other.fqbn;
        }
        return (left.name.replace('/Genuino', '') === other.name.replace('/Genuino', ''));
    }
    Board.sameAs = sameAs;
    function compare(left, right) {
        let result = utils_1.naturalCompare(left.name, right.name);
        if (result === 0) {
            result = utils_1.naturalCompare(left.fqbn || '', right.fqbn || '');
        }
        return result;
    }
    Board.compare = compare;
    function installed(board) {
        return !!board.fqbn;
    }
    Board.installed = installed;
    function toString(board, options = { useFqbn: true }) {
        const fqbn = options && options.useFqbn && board.fqbn ? ` [${board.fqbn}]` : '';
        return `${board.name}${fqbn}`;
    }
    Board.toString = toString;
    function decorateBoards(selectedBoard, boards) {
        // Board names are not unique. We show the corresponding core name as a detail.
        // https://github.com/arduino/arduino-cli/pull/294#issuecomment-513764948
        const distinctBoardNames = new Map();
        for (const { name } of boards) {
            const counter = distinctBoardNames.get(name) || 0;
            distinctBoardNames.set(name, counter + 1);
        }
        // Due to the non-unique board names, we have to check the package name as well.
        const selected = (board) => {
            if (!!selectedBoard) {
                if (Board.equals(board, selectedBoard)) {
                    if ('packageName' in selectedBoard) {
                        return board.packageName === selectedBoard.packageName;
                    }
                    if ('packageId' in selectedBoard) {
                        return board.packageId === selectedBoard.packageId;
                    }
                    return true;
                }
            }
            return false;
        };
        return boards.map((board) => (Object.assign(Object.assign({}, board), { details: (distinctBoardNames.get(board.name) || 0) > 1
                ? ` - ${board.packageName}`
                : undefined, selected: selected(board), missing: !installed(board) })));
    }
    Board.decorateBoards = decorateBoards;
})(Board = exports.Board || (exports.Board = {}));
//# sourceMappingURL=boards-service.js.map