"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SerialMonitorSendInput = void 0;
const React = require("react");
const keys_1 = require("@theia/core/lib/browser/keys");
const boards_service_1 = require("../../../common/protocol/boards-service");
const os_1 = require("@theia/core/lib/common/os");
const common_1 = require("@theia/core/lib/common");
const protocol_1 = require("../plotter/protocol");
class SerialMonitorSendInput extends React.Component {
    constructor(props) {
        super(props);
        this.toDisposeBeforeUnmount = new common_1.DisposableCollection();
        this.setRef = (element) => {
            if (this.props.resolveFocus) {
                this.props.resolveFocus(element || undefined);
            }
        };
        this.state = { text: '', connected: false };
        this.onChange = this.onChange.bind(this);
        this.onSend = this.onSend.bind(this);
        this.onKeyDown = this.onKeyDown.bind(this);
    }
    componentDidMount() {
        this.props.serialConnection.isBESerialConnected().then((connected) => {
            this.setState({ connected });
        });
        this.toDisposeBeforeUnmount.pushAll([
            this.props.serialConnection.onRead(({ messages }) => {
                if (messages.command ===
                    protocol_1.SerialPlotter.Protocol.Command.MIDDLEWARE_CONFIG_CHANGED &&
                    'connected' in messages.data) {
                    this.setState({ connected: messages.data.connected });
                }
            }),
        ]);
    }
    componentWillUnmount() {
        // TODO: "Your preferred browser's local storage is almost full." Discard `content` before saving layout?
        this.toDisposeBeforeUnmount.dispose();
    }
    render() {
        return (React.createElement("input", { ref: this.setRef, type: "text", className: `theia-input ${this.state.connected ? '' : 'warning'}`, placeholder: this.placeholder, value: this.state.text, onChange: this.onChange, onKeyDown: this.onKeyDown }));
    }
    get placeholder() {
        const serialConfig = this.props.serialConnection.getConfig();
        if (!this.state.connected || !serialConfig) {
            return common_1.nls.localize('arduino/serial/notConnected', 'Not connected. Select a board and a port to connect automatically.');
        }
        const { board, port } = serialConfig;
        return common_1.nls.localize('arduino/serial/message', "Message ({0} + Enter to send message to '{1}' on '{2}')", os_1.isOSX ? '⌘' : common_1.nls.localize('vscode/keybindingLabels/ctrlKey', 'Ctrl'), board
            ? boards_service_1.Board.toString(board, {
                useFqbn: false,
            })
            : 'unknown', port ? port.address : 'unknown');
    }
    onChange(event) {
        this.setState({ text: event.target.value });
    }
    onSend() {
        this.props.onSend(this.state.text);
        this.setState({ text: '' });
    }
    onKeyDown(event) {
        const keyCode = keys_1.KeyCode.createKeyCode(event.nativeEvent);
        if (keyCode) {
            const { key, meta, ctrl } = keyCode;
            if (key === keys_1.Key.ENTER && ((os_1.isOSX && meta) || (!os_1.isOSX && ctrl))) {
                this.onSend();
            }
        }
    }
}
exports.SerialMonitorSendInput = SerialMonitorSendInput;
//# sourceMappingURL=serial-monitor-send-input.js.map