"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardsToolBarItem = exports.BoardsDropDown = void 0;
const React = require("react");
const ReactDOM = require("react-dom");
const disposable_1 = require("@theia/core/lib/common/disposable");
const protocol_1 = require("../../common/protocol");
const boards_config_1 = require("./boards-config");
const arduino_commands_1 = require("../arduino-commands");
const boards_service_provider_1 = require("./boards-service-provider");
const common_1 = require("@theia/core/lib/common");
class BoardsDropDown extends React.Component {
    constructor(props) {
        super(props);
        let list = document.getElementById('boards-dropdown-container');
        if (!list) {
            list = document.createElement('div');
            list.id = 'boards-dropdown-container';
            document.body.appendChild(list);
            this.dropdownElement = list;
        }
    }
    render() {
        return ReactDOM.createPortal(this.renderNode(), this.dropdownElement);
    }
    renderNode() {
        const { coords, items } = this.props;
        if (coords === 'hidden') {
            return '';
        }
        const footerLabel = common_1.nls.localize('arduino/board/openBoardsConfig', 'Select other board and port…');
        return (React.createElement("div", { className: "arduino-boards-dropdown-list", style: Object.assign({ position: 'absolute' }, coords) },
            items
                .map(({ name, port, selected, onClick }) => ({
                label: common_1.nls.localize('arduino/board/boardListItem', '{0} at {1}', name, protocol_1.Port.toString(port)),
                selected,
                onClick,
            }))
                .map(this.renderItem),
            React.createElement("div", { key: footerLabel, className: "arduino-boards-dropdown-item arduino-board-dropdown-footer", onClick: () => this.props.openBoardsConfig() },
                React.createElement("div", null, footerLabel))));
    }
    renderItem({ label, selected, onClick, }) {
        return (React.createElement("div", { key: label, className: `arduino-boards-dropdown-item ${selected ? 'selected' : ''}`, onClick: onClick },
            React.createElement("div", null, label),
            selected ? React.createElement("span", { className: "fa fa-check" }) : ''));
    }
}
exports.BoardsDropDown = BoardsDropDown;
class BoardsToolBarItem extends React.Component {
    constructor(props) {
        super(props);
        this.toDispose = new disposable_1.DisposableCollection();
        this.show = (event) => {
            const { currentTarget: element } = event;
            if (element instanceof HTMLElement) {
                if (this.state.coords === 'hidden') {
                    const rect = element.getBoundingClientRect();
                    this.setState({
                        coords: {
                            top: rect.top,
                            left: rect.left,
                            width: rect.width,
                            paddingTop: rect.height,
                        },
                    });
                }
                else {
                    this.setState({ coords: 'hidden' });
                }
            }
            event.stopPropagation();
            event.nativeEvent.stopImmediatePropagation();
        };
        this.openDialog = () => {
            this.props.commands.executeCommand(arduino_commands_1.ArduinoCommands.OPEN_BOARDS_DIALOG.id);
            this.setState({ coords: 'hidden' });
        };
        const { availableBoards } = props.boardsServiceClient;
        this.state = {
            availableBoards,
            coords: 'hidden',
        };
        document.addEventListener('click', () => {
            this.setState({ coords: 'hidden' });
        });
    }
    componentDidMount() {
        this.props.boardsServiceClient.onAvailableBoardsChanged((availableBoards) => this.setState({ availableBoards }));
    }
    componentWillUnmount() {
        this.toDispose.dispose();
    }
    render() {
        const { coords, availableBoards } = this.state;
        const boardsConfig = this.props.boardsServiceClient.boardsConfig;
        const title = boards_config_1.BoardsConfig.Config.toString(boardsConfig, {
            default: common_1.nls.localize('arduino/common/noBoardSelected', 'No board selected'),
        });
        const decorator = (() => {
            const selectedBoard = availableBoards.find(({ selected }) => selected);
            if (!selectedBoard || !selectedBoard.port) {
                return 'fa fa-times notAttached';
            }
            if (selectedBoard.state === boards_service_provider_1.AvailableBoard.State.guessed) {
                return 'fa fa-exclamation-triangle guessed';
            }
            return '';
        })();
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { className: "arduino-boards-toolbar-item-container" },
                React.createElement("div", { className: "arduino-boards-toolbar-item", title: title },
                    React.createElement("div", { className: "inner-container", onClick: this.show },
                        React.createElement("span", { className: decorator }),
                        React.createElement("div", { className: "label noWrapInfo" },
                            React.createElement("div", { className: "noWrapInfo noselect" }, title)),
                        React.createElement("span", { className: "fa fa-caret-down caret" })))),
            React.createElement(BoardsDropDown, { coords: coords, items: availableBoards
                    .filter(boards_service_provider_1.AvailableBoard.hasPort)
                    .map((board) => (Object.assign(Object.assign({}, board), { onClick: () => {
                        if (board.state === boards_service_provider_1.AvailableBoard.State.incomplete) {
                            this.props.boardsServiceClient.boardsConfig = {
                                selectedPort: board.port,
                            };
                            this.openDialog();
                        }
                        else {
                            this.props.boardsServiceClient.boardsConfig = {
                                selectedBoard: board,
                                selectedPort: board.port,
                            };
                        }
                    } }))), openBoardsConfig: this.openDialog })));
    }
}
exports.BoardsToolBarItem = BoardsToolBarItem;
//# sourceMappingURL=boards-toolbar-item.js.map