"use strict";
/********************************************************************************
 * Copyright (C) 2017 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerminalProcess = exports.NodePtyErrors = exports.TerminalProcessFactory = exports.TerminalProcessOptions = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const core_1 = require("@theia/core");
const common_1 = require("@theia/core/lib/common");
const process_1 = require("./process");
const process_manager_1 = require("./process-manager");
const node_pty_1 = require("node-pty");
const multi_ring_buffer_1 = require("./multi-ring-buffer");
const dev_null_stream_1 = require("./dev-null-stream");
const utils_1 = require("./utils");
const pseudo_pty_1 = require("./pseudo-pty");
const stream_1 = require("stream");
exports.TerminalProcessOptions = Symbol('TerminalProcessOptions');
exports.TerminalProcessFactory = Symbol('TerminalProcessFactory');
var NodePtyErrors;
(function (NodePtyErrors) {
    NodePtyErrors["EACCES"] = "Permission denied";
    NodePtyErrors["ENOENT"] = "No such file or directory";
})(NodePtyErrors = exports.NodePtyErrors || (exports.NodePtyErrors = {}));
/**
 * Run arbitrary processes inside pseudo-terminals (PTY).
 *
 * Note: a PTY is not a shell process (bash/pwsh/cmd...)
 */
let TerminalProcess = class TerminalProcess extends process_1.Process {
    constructor(// eslint-disable-next-line @typescript-eslint/indent
    options, processManager, ringBuffer, logger) {
        super(processManager, logger, process_1.ProcessType.Terminal, options);
        this.options = options;
        this.ringBuffer = ringBuffer;
        this.outputStream = this.createOutputStream();
        this.errorStream = new dev_null_stream_1.DevNullStream({ autoDestroy: true });
        if (options.isPseudo) {
            // do not need to spawn a process, new a pseudo pty instead
            this.terminal = new pseudo_pty_1.PseudoPty();
            this.inputStream = new dev_null_stream_1.DevNullStream({ autoDestroy: true });
            return;
        }
        if (this.isForkOptions(this.options)) {
            throw new Error('terminal processes cannot be forked as of today');
        }
        this.logger.debug('Starting terminal process', JSON.stringify(options, undefined, 2));
        try {
            this.terminal = node_pty_1.spawn(options.command, (core_1.isWindows && options.commandLine) || options.args || [], options.options || {});
            process.nextTick(() => this.emitOnStarted());
            // node-pty actually wait for the underlying streams to be closed before emitting exit.
            // We should emulate the `exit` and `close` sequence.
            this.terminal.on('exit', (code, signal) => {
                // Make sure to only pass either code or signal as !undefined, not
                // both.
                //
                // node-pty quirk: On Linux/macOS, if the process exited through the
                // exit syscall (with an exit code), signal will be 0 (an invalid
                // signal value).  If it was terminated because of a signal, the
                // signal parameter will hold the signal number and code should
                // be ignored.
                if (signal === undefined || signal === 0) {
                    this.onTerminalExit(code, undefined);
                }
                else {
                    this.onTerminalExit(undefined, utils_1.signame(signal));
                }
                process.nextTick(() => {
                    if (signal === undefined || signal === 0) {
                        this.emitOnClose(code, undefined);
                    }
                    else {
                        this.emitOnClose(undefined, utils_1.signame(signal));
                    }
                });
            });
            this.terminal.on('data', (data) => {
                ringBuffer.enq(data);
            });
            this.inputStream = new stream_1.Writable({
                write: (chunk) => {
                    this.write(chunk);
                },
            });
        }
        catch (error) {
            this.inputStream = new dev_null_stream_1.DevNullStream({ autoDestroy: true });
            // Normalize the error to make it as close as possible as what
            // node's child_process.spawn would generate in the same
            // situation.
            const message = error.message;
            if (message.startsWith('File not found: ') || message.endsWith(NodePtyErrors.ENOENT)) {
                error.errno = 'ENOENT';
                error.code = 'ENOENT';
                error.path = options.command;
            }
            else if (message.endsWith(NodePtyErrors.EACCES)) {
                error.errno = 'EACCES';
                error.code = 'EACCES';
                error.path = options.command;
            }
            // node-pty throws exceptions on Windows.
            // Call the client error handler, but first give them a chance to register it.
            this.emitOnErrorAsync(error);
        }
    }
    createOutputStream() {
        return this.ringBuffer.getStream();
    }
    get pid() {
        this.checkTerminal();
        return this.terminal.pid;
    }
    get executable() {
        return this.options.command;
    }
    get arguments() {
        return this.options.args || [];
    }
    onTerminalExit(code, signal) {
        this.emitOnExit(code, signal);
        this.unregisterProcess();
    }
    unregisterProcess() {
        this.processManager.unregister(this);
    }
    kill(signal) {
        if (this.terminal && this.killed === false) {
            this.terminal.kill(signal);
        }
    }
    resize(cols, rows) {
        this.checkTerminal();
        this.terminal.resize(cols, rows);
    }
    write(data) {
        this.checkTerminal();
        this.terminal.write(data);
    }
    checkTerminal() {
        if (!this.terminal) {
            throw new Error('pty process did not start correctly');
        }
    }
};
TerminalProcess = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(exports.TerminalProcessOptions)),
    __param(1, inversify_1.inject(process_manager_1.ProcessManager)),
    __param(2, inversify_1.inject(multi_ring_buffer_1.MultiRingBuffer)),
    __param(3, inversify_1.inject(common_1.ILogger)), __param(3, inversify_1.named('process')),
    __metadata("design:paramtypes", [Object, process_manager_1.ProcessManager,
        multi_ring_buffer_1.MultiRingBuffer, Object])
], TerminalProcess);
exports.TerminalProcess = TerminalProcess;
//# sourceMappingURL=terminal-process.js.map