/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { Emitter } from '../../../base/common/event';
import { hash } from '../../../base/common/hash';
import { toDisposable } from '../../../base/common/lifecycle';
import { LRUCache } from '../../../base/common/map';
import { MovingAverage } from '../../../base/common/numbers';
import { score } from './languageSelector';
import { shouldSynchronizeModel } from '../services/modelService';
function isExclusive(selector) {
    if (typeof selector === 'string') {
        return false;
    }
    else if (Array.isArray(selector)) {
        return selector.every(isExclusive);
    }
    else {
        return !!selector.exclusive; // TODO: microsoft/TypeScript#42768
    }
}
var LanguageFeatureRegistry = /** @class */ (function () {
    function LanguageFeatureRegistry() {
        this._clock = 0;
        this._entries = [];
        this._onDidChange = new Emitter();
    }
    Object.defineProperty(LanguageFeatureRegistry.prototype, "onDidChange", {
        get: function () {
            return this._onDidChange.event;
        },
        enumerable: false,
        configurable: true
    });
    LanguageFeatureRegistry.prototype.register = function (selector, provider) {
        var _this = this;
        var entry = {
            selector: selector,
            provider: provider,
            _score: -1,
            _time: this._clock++
        };
        this._entries.push(entry);
        this._lastCandidate = undefined;
        this._onDidChange.fire(this._entries.length);
        return toDisposable(function () {
            if (entry) {
                var idx = _this._entries.indexOf(entry);
                if (idx >= 0) {
                    _this._entries.splice(idx, 1);
                    _this._lastCandidate = undefined;
                    _this._onDidChange.fire(_this._entries.length);
                    entry = undefined;
                }
            }
        });
    };
    LanguageFeatureRegistry.prototype.has = function (model) {
        return this.all(model).length > 0;
    };
    LanguageFeatureRegistry.prototype.all = function (model) {
        var e_1, _a;
        if (!model) {
            return [];
        }
        this._updateScores(model);
        var result = [];
        try {
            // from registry
            for (var _b = __values(this._entries), _c = _b.next(); !_c.done; _c = _b.next()) {
                var entry = _c.value;
                if (entry._score > 0) {
                    result.push(entry.provider);
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return result;
    };
    LanguageFeatureRegistry.prototype.ordered = function (model) {
        var result = [];
        this._orderedForEach(model, function (entry) { return result.push(entry.provider); });
        return result;
    };
    LanguageFeatureRegistry.prototype.orderedGroups = function (model) {
        var result = [];
        var lastBucket;
        var lastBucketScore;
        this._orderedForEach(model, function (entry) {
            if (lastBucket && lastBucketScore === entry._score) {
                lastBucket.push(entry.provider);
            }
            else {
                lastBucketScore = entry._score;
                lastBucket = [entry.provider];
                result.push(lastBucket);
            }
        });
        return result;
    };
    LanguageFeatureRegistry.prototype._orderedForEach = function (model, callback) {
        var e_2, _a;
        if (!model) {
            return;
        }
        this._updateScores(model);
        try {
            for (var _b = __values(this._entries), _c = _b.next(); !_c.done; _c = _b.next()) {
                var entry = _c.value;
                if (entry._score > 0) {
                    callback(entry);
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
    };
    LanguageFeatureRegistry.prototype._updateScores = function (model) {
        var e_3, _a, e_4, _b;
        var candidate = {
            uri: model.uri.toString(),
            language: model.getLanguageIdentifier().language
        };
        if (this._lastCandidate
            && this._lastCandidate.language === candidate.language
            && this._lastCandidate.uri === candidate.uri) {
            // nothing has changed
            return;
        }
        this._lastCandidate = candidate;
        try {
            for (var _c = __values(this._entries), _d = _c.next(); !_d.done; _d = _c.next()) {
                var entry = _d.value;
                entry._score = score(entry.selector, model.uri, model.getLanguageIdentifier().language, shouldSynchronizeModel(model));
                if (isExclusive(entry.selector) && entry._score > 0) {
                    try {
                        // support for one exclusive selector that overwrites
                        // any other selector
                        for (var _e = (e_4 = void 0, __values(this._entries)), _f = _e.next(); !_f.done; _f = _e.next()) {
                            var entry_1 = _f.value;
                            entry_1._score = 0;
                        }
                    }
                    catch (e_4_1) { e_4 = { error: e_4_1 }; }
                    finally {
                        try {
                            if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
                        }
                        finally { if (e_4) throw e_4.error; }
                    }
                    entry._score = 1000;
                    break;
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_3) throw e_3.error; }
        }
        // needs sorting
        this._entries.sort(LanguageFeatureRegistry._compareByScoreAndTime);
    };
    LanguageFeatureRegistry._compareByScoreAndTime = function (a, b) {
        if (a._score < b._score) {
            return 1;
        }
        else if (a._score > b._score) {
            return -1;
        }
        else if (a._time < b._time) {
            return 1;
        }
        else if (a._time > b._time) {
            return -1;
        }
        else {
            return 0;
        }
    };
    return LanguageFeatureRegistry;
}());
export { LanguageFeatureRegistry };
/**
 * Keeps moving average per model and set of providers so that requests
 * can be debounce according to the provider performance
 */
var LanguageFeatureRequestDelays = /** @class */ (function () {
    function LanguageFeatureRequestDelays(_registry, min, max) {
        if (max === void 0) { max = Number.MAX_SAFE_INTEGER; }
        this._registry = _registry;
        this.min = min;
        this.max = max;
        this._cache = new LRUCache(50, 0.7);
    }
    LanguageFeatureRequestDelays.prototype._key = function (model) {
        return model.id + hash(this._registry.all(model));
    };
    LanguageFeatureRequestDelays.prototype._clamp = function (value) {
        if (value === undefined) {
            return this.min;
        }
        else {
            return Math.min(this.max, Math.max(this.min, Math.floor(value * 1.3)));
        }
    };
    LanguageFeatureRequestDelays.prototype.get = function (model) {
        var key = this._key(model);
        var avg = this._cache.get(key);
        return this._clamp(avg === null || avg === void 0 ? void 0 : avg.value);
    };
    LanguageFeatureRequestDelays.prototype.update = function (model, value) {
        var key = this._key(model);
        var avg = this._cache.get(key);
        if (!avg) {
            avg = new MovingAverage();
            this._cache.set(key, avg);
        }
        avg.update(value);
        return this.get(model);
    };
    return LanguageFeatureRequestDelays;
}());
export { LanguageFeatureRequestDelays };
