/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { onUnexpectedError } from '../../../base/common/errors';
import * as strings from '../../../base/common/strings';
import { ReplaceCommand, ReplaceCommandWithOffsetCursorState, ReplaceCommandWithoutChangingPosition, ReplaceCommandThatPreservesSelection } from '../commands/replaceCommand';
import { ShiftCommand } from '../commands/shiftCommand';
import { SurroundSelectionCommand } from '../commands/surroundSelectionCommand';
import { CursorColumns, EditOperationResult, isQuote } from './cursorCommon';
import { getMapForWordSeparators } from './wordCharacterClassifier';
import { Range } from '../core/range';
import { Selection } from '../core/selection';
import { IndentAction } from '../modes/languageConfiguration';
import { LanguageConfigurationRegistry } from '../modes/languageConfigurationRegistry';
var TypeOperations = /** @class */ (function () {
    function TypeOperations() {
    }
    TypeOperations.indent = function (config, model, selections) {
        if (model === null || selections === null) {
            return [];
        }
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = new ShiftCommand(selections[i], {
                isUnshift: false,
                tabSize: config.tabSize,
                indentSize: config.indentSize,
                insertSpaces: config.insertSpaces,
                useTabStops: config.useTabStops,
                autoIndent: config.autoIndent
            });
        }
        return commands;
    };
    TypeOperations.outdent = function (config, model, selections) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = new ShiftCommand(selections[i], {
                isUnshift: true,
                tabSize: config.tabSize,
                indentSize: config.indentSize,
                insertSpaces: config.insertSpaces,
                useTabStops: config.useTabStops,
                autoIndent: config.autoIndent
            });
        }
        return commands;
    };
    TypeOperations.shiftIndent = function (config, indentation, count) {
        count = count || 1;
        return ShiftCommand.shiftIndent(indentation, indentation.length + count, config.tabSize, config.indentSize, config.insertSpaces);
    };
    TypeOperations.unshiftIndent = function (config, indentation, count) {
        count = count || 1;
        return ShiftCommand.unshiftIndent(indentation, indentation.length + count, config.tabSize, config.indentSize, config.insertSpaces);
    };
    TypeOperations._distributedPaste = function (config, model, selections, text) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = new ReplaceCommand(selections[i], text[i]);
        }
        return new EditOperationResult(0 /* Other */, commands, {
            shouldPushStackElementBefore: true,
            shouldPushStackElementAfter: true
        });
    };
    TypeOperations._simplePaste = function (config, model, selections, text, pasteOnNewLine) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            var position = selection.getPosition();
            if (pasteOnNewLine && !selection.isEmpty()) {
                pasteOnNewLine = false;
            }
            if (pasteOnNewLine && text.indexOf('\n') !== text.length - 1) {
                pasteOnNewLine = false;
            }
            if (pasteOnNewLine) {
                // Paste entire line at the beginning of line
                var typeSelection = new Range(position.lineNumber, 1, position.lineNumber, 1);
                commands[i] = new ReplaceCommandThatPreservesSelection(typeSelection, text, selection, true);
            }
            else {
                commands[i] = new ReplaceCommand(selection, text);
            }
        }
        return new EditOperationResult(0 /* Other */, commands, {
            shouldPushStackElementBefore: true,
            shouldPushStackElementAfter: true
        });
    };
    TypeOperations._distributePasteToCursors = function (config, selections, text, pasteOnNewLine, multicursorText) {
        if (pasteOnNewLine) {
            return null;
        }
        if (selections.length === 1) {
            return null;
        }
        if (multicursorText && multicursorText.length === selections.length) {
            return multicursorText;
        }
        if (config.multiCursorPaste === 'spread') {
            // Try to spread the pasted text in case the line count matches the cursor count
            // Remove trailing \n if present
            if (text.charCodeAt(text.length - 1) === 10 /* LineFeed */) {
                text = text.substr(0, text.length - 1);
            }
            // Remove trailing \r if present
            if (text.charCodeAt(text.length - 1) === 13 /* CarriageReturn */) {
                text = text.substr(0, text.length - 1);
            }
            var lines = strings.splitLines(text);
            if (lines.length === selections.length) {
                return lines;
            }
        }
        return null;
    };
    TypeOperations.paste = function (config, model, selections, text, pasteOnNewLine, multicursorText) {
        var distributedPaste = this._distributePasteToCursors(config, selections, text, pasteOnNewLine, multicursorText);
        if (distributedPaste) {
            selections = selections.sort(Range.compareRangesUsingStarts);
            return this._distributedPaste(config, model, selections, distributedPaste);
        }
        else {
            return this._simplePaste(config, model, selections, text, pasteOnNewLine);
        }
    };
    TypeOperations._goodIndentForLine = function (config, model, lineNumber) {
        var action = null;
        var indentation = '';
        var expectedIndentAction = LanguageConfigurationRegistry.getInheritIndentForLine(config.autoIndent, model, lineNumber, false);
        if (expectedIndentAction) {
            action = expectedIndentAction.action;
            indentation = expectedIndentAction.indentation;
        }
        else if (lineNumber > 1) {
            var lastLineNumber = void 0;
            for (lastLineNumber = lineNumber - 1; lastLineNumber >= 1; lastLineNumber--) {
                var lineText = model.getLineContent(lastLineNumber);
                var nonWhitespaceIdx = strings.lastNonWhitespaceIndex(lineText);
                if (nonWhitespaceIdx >= 0) {
                    break;
                }
            }
            if (lastLineNumber < 1) {
                // No previous line with content found
                return null;
            }
            var maxColumn = model.getLineMaxColumn(lastLineNumber);
            var expectedEnterAction = LanguageConfigurationRegistry.getEnterAction(config.autoIndent, model, new Range(lastLineNumber, maxColumn, lastLineNumber, maxColumn));
            if (expectedEnterAction) {
                indentation = expectedEnterAction.indentation + expectedEnterAction.appendText;
            }
        }
        if (action) {
            if (action === IndentAction.Indent) {
                indentation = TypeOperations.shiftIndent(config, indentation);
            }
            if (action === IndentAction.Outdent) {
                indentation = TypeOperations.unshiftIndent(config, indentation);
            }
            indentation = config.normalizeIndentation(indentation);
        }
        if (!indentation) {
            return null;
        }
        return indentation;
    };
    TypeOperations._replaceJumpToNextIndent = function (config, model, selection, insertsAutoWhitespace) {
        var typeText = '';
        var position = selection.getStartPosition();
        if (config.insertSpaces) {
            var visibleColumnFromColumn = CursorColumns.visibleColumnFromColumn2(config, model, position);
            var indentSize = config.indentSize;
            var spacesCnt = indentSize - (visibleColumnFromColumn % indentSize);
            for (var i = 0; i < spacesCnt; i++) {
                typeText += ' ';
            }
        }
        else {
            typeText = '\t';
        }
        return new ReplaceCommand(selection, typeText, insertsAutoWhitespace);
    };
    TypeOperations.tab = function (config, model, selections) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            if (selection.isEmpty()) {
                var lineText = model.getLineContent(selection.startLineNumber);
                if (/^\s*$/.test(lineText) && model.isCheapToTokenize(selection.startLineNumber)) {
                    var goodIndent = this._goodIndentForLine(config, model, selection.startLineNumber);
                    goodIndent = goodIndent || '\t';
                    var possibleTypeText = config.normalizeIndentation(goodIndent);
                    if (!lineText.startsWith(possibleTypeText)) {
                        commands[i] = new ReplaceCommand(new Range(selection.startLineNumber, 1, selection.startLineNumber, lineText.length + 1), possibleTypeText, true);
                        continue;
                    }
                }
                commands[i] = this._replaceJumpToNextIndent(config, model, selection, true);
            }
            else {
                if (selection.startLineNumber === selection.endLineNumber) {
                    var lineMaxColumn = model.getLineMaxColumn(selection.startLineNumber);
                    if (selection.startColumn !== 1 || selection.endColumn !== lineMaxColumn) {
                        // This is a single line selection that is not the entire line
                        commands[i] = this._replaceJumpToNextIndent(config, model, selection, false);
                        continue;
                    }
                }
                commands[i] = new ShiftCommand(selection, {
                    isUnshift: false,
                    tabSize: config.tabSize,
                    indentSize: config.indentSize,
                    insertSpaces: config.insertSpaces,
                    useTabStops: config.useTabStops,
                    autoIndent: config.autoIndent
                });
            }
        }
        return commands;
    };
    TypeOperations.compositionType = function (prevEditOperationType, config, model, selections, text, replacePrevCharCnt, replaceNextCharCnt, positionDelta) {
        var _this = this;
        var commands = selections.map(function (selection) { return _this._compositionType(model, selection, text, replacePrevCharCnt, replaceNextCharCnt, positionDelta); });
        return new EditOperationResult(1 /* Typing */, commands, {
            shouldPushStackElementBefore: (prevEditOperationType !== 1 /* Typing */),
            shouldPushStackElementAfter: false
        });
    };
    TypeOperations._compositionType = function (model, selection, text, replacePrevCharCnt, replaceNextCharCnt, positionDelta) {
        if (!selection.isEmpty()) {
            // looks like https://github.com/microsoft/vscode/issues/2773
            // where a cursor operation occurred before a canceled composition
            // => ignore composition
            return null;
        }
        var pos = selection.getPosition();
        var startColumn = Math.max(1, pos.column - replacePrevCharCnt);
        var endColumn = Math.min(model.getLineMaxColumn(pos.lineNumber), pos.column + replaceNextCharCnt);
        var range = new Range(pos.lineNumber, startColumn, pos.lineNumber, endColumn);
        var oldText = model.getValueInRange(range);
        if (oldText === text && positionDelta === 0) {
            // => ignore composition that doesn't do anything
            return null;
        }
        return new ReplaceCommandWithOffsetCursorState(range, text, 0, positionDelta);
    };
    TypeOperations._typeCommand = function (range, text, keepPosition) {
        if (keepPosition) {
            return new ReplaceCommandWithoutChangingPosition(range, text, true);
        }
        else {
            return new ReplaceCommand(range, text, true);
        }
    };
    TypeOperations._enter = function (config, model, keepPosition, range) {
        if (config.autoIndent === 0 /* None */) {
            return TypeOperations._typeCommand(range, '\n', keepPosition);
        }
        if (!model.isCheapToTokenize(range.getStartPosition().lineNumber) || config.autoIndent === 1 /* Keep */) {
            var lineText_1 = model.getLineContent(range.startLineNumber);
            var indentation_1 = strings.getLeadingWhitespace(lineText_1).substring(0, range.startColumn - 1);
            return TypeOperations._typeCommand(range, '\n' + config.normalizeIndentation(indentation_1), keepPosition);
        }
        var r = LanguageConfigurationRegistry.getEnterAction(config.autoIndent, model, range);
        if (r) {
            if (r.indentAction === IndentAction.None) {
                // Nothing special
                return TypeOperations._typeCommand(range, '\n' + config.normalizeIndentation(r.indentation + r.appendText), keepPosition);
            }
            else if (r.indentAction === IndentAction.Indent) {
                // Indent once
                return TypeOperations._typeCommand(range, '\n' + config.normalizeIndentation(r.indentation + r.appendText), keepPosition);
            }
            else if (r.indentAction === IndentAction.IndentOutdent) {
                // Ultra special
                var normalIndent = config.normalizeIndentation(r.indentation);
                var increasedIndent = config.normalizeIndentation(r.indentation + r.appendText);
                var typeText = '\n' + increasedIndent + '\n' + normalIndent;
                if (keepPosition) {
                    return new ReplaceCommandWithoutChangingPosition(range, typeText, true);
                }
                else {
                    return new ReplaceCommandWithOffsetCursorState(range, typeText, -1, increasedIndent.length - normalIndent.length, true);
                }
            }
            else if (r.indentAction === IndentAction.Outdent) {
                var actualIndentation = TypeOperations.unshiftIndent(config, r.indentation);
                return TypeOperations._typeCommand(range, '\n' + config.normalizeIndentation(actualIndentation + r.appendText), keepPosition);
            }
        }
        var lineText = model.getLineContent(range.startLineNumber);
        var indentation = strings.getLeadingWhitespace(lineText).substring(0, range.startColumn - 1);
        if (config.autoIndent >= 4 /* Full */) {
            var ir = LanguageConfigurationRegistry.getIndentForEnter(config.autoIndent, model, range, {
                unshiftIndent: function (indent) {
                    return TypeOperations.unshiftIndent(config, indent);
                },
                shiftIndent: function (indent) {
                    return TypeOperations.shiftIndent(config, indent);
                },
                normalizeIndentation: function (indent) {
                    return config.normalizeIndentation(indent);
                }
            });
            if (ir) {
                var oldEndViewColumn = CursorColumns.visibleColumnFromColumn2(config, model, range.getEndPosition());
                var oldEndColumn = range.endColumn;
                var newLineContent = model.getLineContent(range.endLineNumber);
                var firstNonWhitespace = strings.firstNonWhitespaceIndex(newLineContent);
                if (firstNonWhitespace >= 0) {
                    range = range.setEndPosition(range.endLineNumber, Math.max(range.endColumn, firstNonWhitespace + 1));
                }
                else {
                    range = range.setEndPosition(range.endLineNumber, model.getLineMaxColumn(range.endLineNumber));
                }
                if (keepPosition) {
                    return new ReplaceCommandWithoutChangingPosition(range, '\n' + config.normalizeIndentation(ir.afterEnter), true);
                }
                else {
                    var offset = 0;
                    if (oldEndColumn <= firstNonWhitespace + 1) {
                        if (!config.insertSpaces) {
                            oldEndViewColumn = Math.ceil(oldEndViewColumn / config.indentSize);
                        }
                        offset = Math.min(oldEndViewColumn + 1 - config.normalizeIndentation(ir.afterEnter).length - 1, 0);
                    }
                    return new ReplaceCommandWithOffsetCursorState(range, '\n' + config.normalizeIndentation(ir.afterEnter), 0, offset, true);
                }
            }
        }
        return TypeOperations._typeCommand(range, '\n' + config.normalizeIndentation(indentation), keepPosition);
    };
    TypeOperations._isAutoIndentType = function (config, model, selections) {
        if (config.autoIndent < 4 /* Full */) {
            return false;
        }
        for (var i = 0, len = selections.length; i < len; i++) {
            if (!model.isCheapToTokenize(selections[i].getEndPosition().lineNumber)) {
                return false;
            }
        }
        return true;
    };
    TypeOperations._runAutoIndentType = function (config, model, range, ch) {
        var currentIndentation = LanguageConfigurationRegistry.getIndentationAtPosition(model, range.startLineNumber, range.startColumn);
        var actualIndentation = LanguageConfigurationRegistry.getIndentActionForType(config.autoIndent, model, range, ch, {
            shiftIndent: function (indentation) {
                return TypeOperations.shiftIndent(config, indentation);
            },
            unshiftIndent: function (indentation) {
                return TypeOperations.unshiftIndent(config, indentation);
            },
        });
        if (actualIndentation === null) {
            return null;
        }
        if (actualIndentation !== config.normalizeIndentation(currentIndentation)) {
            var firstNonWhitespace = model.getLineFirstNonWhitespaceColumn(range.startLineNumber);
            if (firstNonWhitespace === 0) {
                return TypeOperations._typeCommand(new Range(range.startLineNumber, 1, range.endLineNumber, range.endColumn), config.normalizeIndentation(actualIndentation) + ch, false);
            }
            else {
                return TypeOperations._typeCommand(new Range(range.startLineNumber, 1, range.endLineNumber, range.endColumn), config.normalizeIndentation(actualIndentation) +
                    model.getLineContent(range.startLineNumber).substring(firstNonWhitespace - 1, range.startColumn - 1) + ch, false);
            }
        }
        return null;
    };
    TypeOperations._isAutoClosingOvertype = function (config, model, selections, autoClosedCharacters, ch) {
        if (config.autoClosingOvertype === 'never') {
            return false;
        }
        if (!config.autoClosingPairs.autoClosingPairsCloseSingleChar.has(ch)) {
            return false;
        }
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            if (!selection.isEmpty()) {
                return false;
            }
            var position = selection.getPosition();
            var lineText = model.getLineContent(position.lineNumber);
            var afterCharacter = lineText.charAt(position.column - 1);
            if (afterCharacter !== ch) {
                return false;
            }
            // Do not over-type quotes after a backslash
            var chIsQuote = isQuote(ch);
            var beforeCharacter = position.column > 2 ? lineText.charCodeAt(position.column - 2) : 0 /* Null */;
            if (beforeCharacter === 92 /* Backslash */ && chIsQuote) {
                return false;
            }
            // Must over-type a closing character typed by the editor
            if (config.autoClosingOvertype === 'auto') {
                var found = false;
                for (var j = 0, lenJ = autoClosedCharacters.length; j < lenJ; j++) {
                    var autoClosedCharacter = autoClosedCharacters[j];
                    if (position.lineNumber === autoClosedCharacter.startLineNumber && position.column === autoClosedCharacter.startColumn) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    return false;
                }
            }
        }
        return true;
    };
    TypeOperations._runAutoClosingOvertype = function (prevEditOperationType, config, model, selections, ch) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            var position = selection.getPosition();
            var typeSelection = new Range(position.lineNumber, position.column, position.lineNumber, position.column + 1);
            commands[i] = new ReplaceCommand(typeSelection, ch);
        }
        return new EditOperationResult(1 /* Typing */, commands, {
            shouldPushStackElementBefore: (prevEditOperationType !== 1 /* Typing */),
            shouldPushStackElementAfter: false
        });
    };
    TypeOperations._isBeforeClosingBrace = function (config, lineAfter) {
        // If the start of lineAfter can be interpretted as both a starting or ending brace, default to returning false
        var nextChar = lineAfter.charAt(0);
        var potentialStartingBraces = config.autoClosingPairs.autoClosingPairsOpenByStart.get(nextChar) || [];
        var potentialClosingBraces = config.autoClosingPairs.autoClosingPairsCloseByStart.get(nextChar) || [];
        var isBeforeStartingBrace = potentialStartingBraces.some(function (x) { return lineAfter.startsWith(x.open); });
        var isBeforeClosingBrace = potentialClosingBraces.some(function (x) { return lineAfter.startsWith(x.close); });
        return !isBeforeStartingBrace && isBeforeClosingBrace;
    };
    TypeOperations._findAutoClosingPairOpen = function (config, model, positions, ch) {
        var e_1, _a, e_2, _b;
        var autoClosingPairCandidates = config.autoClosingPairs.autoClosingPairsOpenByEnd.get(ch);
        if (!autoClosingPairCandidates) {
            return null;
        }
        // Determine which auto-closing pair it is
        var autoClosingPair = null;
        try {
            for (var autoClosingPairCandidates_1 = __values(autoClosingPairCandidates), autoClosingPairCandidates_1_1 = autoClosingPairCandidates_1.next(); !autoClosingPairCandidates_1_1.done; autoClosingPairCandidates_1_1 = autoClosingPairCandidates_1.next()) {
                var autoClosingPairCandidate = autoClosingPairCandidates_1_1.value;
                if (autoClosingPair === null || autoClosingPairCandidate.open.length > autoClosingPair.open.length) {
                    var candidateIsMatch = true;
                    try {
                        for (var positions_1 = (e_2 = void 0, __values(positions)), positions_1_1 = positions_1.next(); !positions_1_1.done; positions_1_1 = positions_1.next()) {
                            var position = positions_1_1.value;
                            var relevantText = model.getValueInRange(new Range(position.lineNumber, position.column - autoClosingPairCandidate.open.length + 1, position.lineNumber, position.column));
                            if (relevantText + ch !== autoClosingPairCandidate.open) {
                                candidateIsMatch = false;
                                break;
                            }
                        }
                    }
                    catch (e_2_1) { e_2 = { error: e_2_1 }; }
                    finally {
                        try {
                            if (positions_1_1 && !positions_1_1.done && (_b = positions_1.return)) _b.call(positions_1);
                        }
                        finally { if (e_2) throw e_2.error; }
                    }
                    if (candidateIsMatch) {
                        autoClosingPair = autoClosingPairCandidate;
                    }
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (autoClosingPairCandidates_1_1 && !autoClosingPairCandidates_1_1.done && (_a = autoClosingPairCandidates_1.return)) _a.call(autoClosingPairCandidates_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return autoClosingPair;
    };
    TypeOperations._findSubAutoClosingPairClose = function (config, autoClosingPair) {
        var e_3, _a;
        if (autoClosingPair.open.length <= 1) {
            return '';
        }
        var lastChar = autoClosingPair.close.charAt(autoClosingPair.close.length - 1);
        // get candidates with the same last character as close
        var subPairCandidates = config.autoClosingPairs.autoClosingPairsCloseByEnd.get(lastChar) || [];
        var subPairMatch = null;
        try {
            for (var subPairCandidates_1 = __values(subPairCandidates), subPairCandidates_1_1 = subPairCandidates_1.next(); !subPairCandidates_1_1.done; subPairCandidates_1_1 = subPairCandidates_1.next()) {
                var x = subPairCandidates_1_1.value;
                if (x.open !== autoClosingPair.open && autoClosingPair.open.includes(x.open) && autoClosingPair.close.endsWith(x.close)) {
                    if (!subPairMatch || x.open.length > subPairMatch.open.length) {
                        subPairMatch = x;
                    }
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (subPairCandidates_1_1 && !subPairCandidates_1_1.done && (_a = subPairCandidates_1.return)) _a.call(subPairCandidates_1);
            }
            finally { if (e_3) throw e_3.error; }
        }
        if (subPairMatch) {
            return subPairMatch.close;
        }
        else {
            return '';
        }
    };
    TypeOperations._getAutoClosingPairClose = function (config, model, selections, ch, insertOpenCharacter) {
        var chIsQuote = isQuote(ch);
        var autoCloseConfig = chIsQuote ? config.autoClosingQuotes : config.autoClosingBrackets;
        if (autoCloseConfig === 'never') {
            return null;
        }
        var autoClosingPair = this._findAutoClosingPairOpen(config, model, selections.map(function (s) { return s.getPosition(); }), ch);
        if (!autoClosingPair) {
            return null;
        }
        var subAutoClosingPairClose = this._findSubAutoClosingPairClose(config, autoClosingPair);
        var isSubAutoClosingPairPresent = true;
        var shouldAutoCloseBefore = chIsQuote ? config.shouldAutoCloseBefore.quote : config.shouldAutoCloseBefore.bracket;
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            if (!selection.isEmpty()) {
                return null;
            }
            var position = selection.getPosition();
            var lineText = model.getLineContent(position.lineNumber);
            var lineAfter = lineText.substring(position.column - 1);
            if (!lineAfter.startsWith(subAutoClosingPairClose)) {
                isSubAutoClosingPairPresent = false;
            }
            // Only consider auto closing the pair if an allowed character follows or if another autoclosed pair closing brace follows
            if (lineText.length > position.column - 1) {
                var characterAfter = lineText.charAt(position.column - 1);
                var isBeforeCloseBrace = TypeOperations._isBeforeClosingBrace(config, lineAfter);
                if (!isBeforeCloseBrace && !shouldAutoCloseBefore(characterAfter)) {
                    return null;
                }
            }
            if (!model.isCheapToTokenize(position.lineNumber)) {
                // Do not force tokenization
                return null;
            }
            // Do not auto-close ' or " after a word character
            if (autoClosingPair.open.length === 1 && chIsQuote && autoCloseConfig !== 'always') {
                var wordSeparators = getMapForWordSeparators(config.wordSeparators);
                if (insertOpenCharacter && position.column > 1 && wordSeparators.get(lineText.charCodeAt(position.column - 2)) === 0 /* Regular */) {
                    return null;
                }
                if (!insertOpenCharacter && position.column > 2 && wordSeparators.get(lineText.charCodeAt(position.column - 3)) === 0 /* Regular */) {
                    return null;
                }
            }
            model.forceTokenization(position.lineNumber);
            var lineTokens = model.getLineTokens(position.lineNumber);
            var shouldAutoClosePair = false;
            try {
                shouldAutoClosePair = LanguageConfigurationRegistry.shouldAutoClosePair(autoClosingPair, lineTokens, insertOpenCharacter ? position.column : position.column - 1);
            }
            catch (e) {
                onUnexpectedError(e);
            }
            if (!shouldAutoClosePair) {
                return null;
            }
        }
        if (isSubAutoClosingPairPresent) {
            return autoClosingPair.close.substring(0, autoClosingPair.close.length - subAutoClosingPairClose.length);
        }
        else {
            return autoClosingPair.close;
        }
    };
    TypeOperations._runAutoClosingOpenCharType = function (prevEditOperationType, config, model, selections, ch, insertOpenCharacter, autoClosingPairClose) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            commands[i] = new TypeWithAutoClosingCommand(selection, ch, insertOpenCharacter, autoClosingPairClose);
        }
        return new EditOperationResult(1 /* Typing */, commands, {
            shouldPushStackElementBefore: true,
            shouldPushStackElementAfter: false
        });
    };
    TypeOperations._shouldSurroundChar = function (config, ch) {
        if (isQuote(ch)) {
            return (config.autoSurround === 'quotes' || config.autoSurround === 'languageDefined');
        }
        else {
            // Character is a bracket
            return (config.autoSurround === 'brackets' || config.autoSurround === 'languageDefined');
        }
    };
    TypeOperations._isSurroundSelectionType = function (config, model, selections, ch) {
        if (!TypeOperations._shouldSurroundChar(config, ch) || !config.surroundingPairs.hasOwnProperty(ch)) {
            return false;
        }
        var isTypingAQuoteCharacter = isQuote(ch);
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            if (selection.isEmpty()) {
                return false;
            }
            var selectionContainsOnlyWhitespace = true;
            for (var lineNumber = selection.startLineNumber; lineNumber <= selection.endLineNumber; lineNumber++) {
                var lineText = model.getLineContent(lineNumber);
                var startIndex = (lineNumber === selection.startLineNumber ? selection.startColumn - 1 : 0);
                var endIndex = (lineNumber === selection.endLineNumber ? selection.endColumn - 1 : lineText.length);
                var selectedText = lineText.substring(startIndex, endIndex);
                if (/[^ \t]/.test(selectedText)) {
                    // this selected text contains something other than whitespace
                    selectionContainsOnlyWhitespace = false;
                    break;
                }
            }
            if (selectionContainsOnlyWhitespace) {
                return false;
            }
            if (isTypingAQuoteCharacter && selection.startLineNumber === selection.endLineNumber && selection.startColumn + 1 === selection.endColumn) {
                var selectionText = model.getValueInRange(selection);
                if (isQuote(selectionText)) {
                    // Typing a quote character on top of another quote character
                    // => disable surround selection type
                    return false;
                }
            }
        }
        return true;
    };
    TypeOperations._runSurroundSelectionType = function (prevEditOperationType, config, model, selections, ch) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            var closeCharacter = config.surroundingPairs[ch];
            commands[i] = new SurroundSelectionCommand(selection, ch, closeCharacter);
        }
        return new EditOperationResult(0 /* Other */, commands, {
            shouldPushStackElementBefore: true,
            shouldPushStackElementAfter: true
        });
    };
    TypeOperations._isTypeInterceptorElectricChar = function (config, model, selections) {
        if (selections.length === 1 && model.isCheapToTokenize(selections[0].getEndPosition().lineNumber)) {
            return true;
        }
        return false;
    };
    TypeOperations._typeInterceptorElectricChar = function (prevEditOperationType, config, model, selection, ch) {
        if (!config.electricChars.hasOwnProperty(ch) || !selection.isEmpty()) {
            return null;
        }
        var position = selection.getPosition();
        model.forceTokenization(position.lineNumber);
        var lineTokens = model.getLineTokens(position.lineNumber);
        var electricAction;
        try {
            electricAction = LanguageConfigurationRegistry.onElectricCharacter(ch, lineTokens, position.column);
        }
        catch (e) {
            onUnexpectedError(e);
            return null;
        }
        if (!electricAction) {
            return null;
        }
        if (electricAction.matchOpenBracket) {
            var endColumn = (lineTokens.getLineContent() + ch).lastIndexOf(electricAction.matchOpenBracket) + 1;
            var match = model.findMatchingBracketUp(electricAction.matchOpenBracket, {
                lineNumber: position.lineNumber,
                column: endColumn
            });
            if (match) {
                if (match.startLineNumber === position.lineNumber) {
                    // matched something on the same line => no change in indentation
                    return null;
                }
                var matchLine = model.getLineContent(match.startLineNumber);
                var matchLineIndentation = strings.getLeadingWhitespace(matchLine);
                var newIndentation = config.normalizeIndentation(matchLineIndentation);
                var lineText = model.getLineContent(position.lineNumber);
                var lineFirstNonBlankColumn = model.getLineFirstNonWhitespaceColumn(position.lineNumber) || position.column;
                var prefix = lineText.substring(lineFirstNonBlankColumn - 1, position.column - 1);
                var typeText = newIndentation + prefix + ch;
                var typeSelection = new Range(position.lineNumber, 1, position.lineNumber, position.column);
                var command = new ReplaceCommand(typeSelection, typeText);
                return new EditOperationResult(1 /* Typing */, [command], {
                    shouldPushStackElementBefore: false,
                    shouldPushStackElementAfter: true
                });
            }
        }
        return null;
    };
    /**
     * This is very similar with typing, but the character is already in the text buffer!
     */
    TypeOperations.compositionEndWithInterceptors = function (prevEditOperationType, config, model, selectionsWhenCompositionStarted, selections, autoClosedCharacters) {
        var e_4, _a;
        if (!selectionsWhenCompositionStarted || Selection.selectionsArrEqual(selectionsWhenCompositionStarted, selections)) {
            // no content was typed
            return null;
        }
        var ch = null;
        try {
            // extract last typed character
            for (var selections_1 = __values(selections), selections_1_1 = selections_1.next(); !selections_1_1.done; selections_1_1 = selections_1.next()) {
                var selection = selections_1_1.value;
                if (!selection.isEmpty()) {
                    return null;
                }
                var position = selection.getPosition();
                var currentChar = model.getValueInRange(new Range(position.lineNumber, position.column - 1, position.lineNumber, position.column));
                if (ch === null) {
                    ch = currentChar;
                }
                else if (ch !== currentChar) {
                    return null;
                }
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (selections_1_1 && !selections_1_1.done && (_a = selections_1.return)) _a.call(selections_1);
            }
            finally { if (e_4) throw e_4.error; }
        }
        if (!ch) {
            return null;
        }
        if (this._isAutoClosingOvertype(config, model, selections, autoClosedCharacters, ch)) {
            // Unfortunately, the close character is at this point "doubled", so we need to delete it...
            var commands = selections.map(function (s) { return new ReplaceCommand(new Range(s.positionLineNumber, s.positionColumn, s.positionLineNumber, s.positionColumn + 1), '', false); });
            return new EditOperationResult(1 /* Typing */, commands, {
                shouldPushStackElementBefore: true,
                shouldPushStackElementAfter: false
            });
        }
        var autoClosingPairClose = this._getAutoClosingPairClose(config, model, selections, ch, false);
        if (autoClosingPairClose !== null) {
            return this._runAutoClosingOpenCharType(prevEditOperationType, config, model, selections, ch, false, autoClosingPairClose);
        }
        return null;
    };
    TypeOperations.typeWithInterceptors = function (isDoingComposition, prevEditOperationType, config, model, selections, autoClosedCharacters, ch) {
        if (!isDoingComposition && ch === '\n') {
            var commands_1 = [];
            for (var i = 0, len = selections.length; i < len; i++) {
                commands_1[i] = TypeOperations._enter(config, model, false, selections[i]);
            }
            return new EditOperationResult(1 /* Typing */, commands_1, {
                shouldPushStackElementBefore: true,
                shouldPushStackElementAfter: false,
            });
        }
        if (!isDoingComposition && this._isAutoIndentType(config, model, selections)) {
            var commands_2 = [];
            var autoIndentFails = false;
            for (var i = 0, len = selections.length; i < len; i++) {
                commands_2[i] = this._runAutoIndentType(config, model, selections[i], ch);
                if (!commands_2[i]) {
                    autoIndentFails = true;
                    break;
                }
            }
            if (!autoIndentFails) {
                return new EditOperationResult(1 /* Typing */, commands_2, {
                    shouldPushStackElementBefore: true,
                    shouldPushStackElementAfter: false,
                });
            }
        }
        if (!isDoingComposition && this._isAutoClosingOvertype(config, model, selections, autoClosedCharacters, ch)) {
            return this._runAutoClosingOvertype(prevEditOperationType, config, model, selections, ch);
        }
        if (!isDoingComposition) {
            var autoClosingPairClose = this._getAutoClosingPairClose(config, model, selections, ch, true);
            if (autoClosingPairClose) {
                return this._runAutoClosingOpenCharType(prevEditOperationType, config, model, selections, ch, true, autoClosingPairClose);
            }
        }
        if (this._isSurroundSelectionType(config, model, selections, ch)) {
            return this._runSurroundSelectionType(prevEditOperationType, config, model, selections, ch);
        }
        // Electric characters make sense only when dealing with a single cursor,
        // as multiple cursors typing brackets for example would interfer with bracket matching
        if (!isDoingComposition && this._isTypeInterceptorElectricChar(config, model, selections)) {
            var r = this._typeInterceptorElectricChar(prevEditOperationType, config, model, selections[0], ch);
            if (r) {
                return r;
            }
        }
        // A simple character type
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = new ReplaceCommand(selections[i], ch);
        }
        var shouldPushStackElementBefore = (prevEditOperationType !== 1 /* Typing */);
        if (ch === ' ') {
            shouldPushStackElementBefore = true;
        }
        return new EditOperationResult(1 /* Typing */, commands, {
            shouldPushStackElementBefore: shouldPushStackElementBefore,
            shouldPushStackElementAfter: false
        });
    };
    TypeOperations.typeWithoutInterceptors = function (prevEditOperationType, config, model, selections, str) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = new ReplaceCommand(selections[i], str);
        }
        return new EditOperationResult(1 /* Typing */, commands, {
            shouldPushStackElementBefore: (prevEditOperationType !== 1 /* Typing */),
            shouldPushStackElementAfter: false
        });
    };
    TypeOperations.lineInsertBefore = function (config, model, selections) {
        if (model === null || selections === null) {
            return [];
        }
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var lineNumber = selections[i].positionLineNumber;
            if (lineNumber === 1) {
                commands[i] = new ReplaceCommandWithoutChangingPosition(new Range(1, 1, 1, 1), '\n');
            }
            else {
                lineNumber--;
                var column = model.getLineMaxColumn(lineNumber);
                commands[i] = this._enter(config, model, false, new Range(lineNumber, column, lineNumber, column));
            }
        }
        return commands;
    };
    TypeOperations.lineInsertAfter = function (config, model, selections) {
        if (model === null || selections === null) {
            return [];
        }
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            var lineNumber = selections[i].positionLineNumber;
            var column = model.getLineMaxColumn(lineNumber);
            commands[i] = this._enter(config, model, false, new Range(lineNumber, column, lineNumber, column));
        }
        return commands;
    };
    TypeOperations.lineBreakInsert = function (config, model, selections) {
        var commands = [];
        for (var i = 0, len = selections.length; i < len; i++) {
            commands[i] = this._enter(config, model, true, selections[i]);
        }
        return commands;
    };
    return TypeOperations;
}());
export { TypeOperations };
var TypeWithAutoClosingCommand = /** @class */ (function (_super) {
    __extends(TypeWithAutoClosingCommand, _super);
    function TypeWithAutoClosingCommand(selection, openCharacter, insertOpenCharacter, closeCharacter) {
        var _this = _super.call(this, selection, (insertOpenCharacter ? openCharacter : '') + closeCharacter, 0, -closeCharacter.length) || this;
        _this._openCharacter = openCharacter;
        _this._closeCharacter = closeCharacter;
        _this.closeCharacterRange = null;
        _this.enclosingRange = null;
        return _this;
    }
    TypeWithAutoClosingCommand.prototype.computeCursorState = function (model, helper) {
        var inverseEditOperations = helper.getInverseEditOperations();
        var range = inverseEditOperations[0].range;
        this.closeCharacterRange = new Range(range.startLineNumber, range.endColumn - this._closeCharacter.length, range.endLineNumber, range.endColumn);
        this.enclosingRange = new Range(range.startLineNumber, range.endColumn - this._openCharacter.length - this._closeCharacter.length, range.endLineNumber, range.endColumn);
        return _super.prototype.computeCursorState.call(this, model, helper);
    };
    return TypeWithAutoClosingCommand;
}(ReplaceCommandWithOffsetCursorState));
export { TypeWithAutoClosingCommand };
