/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { Range } from '../../../common/range';
/**
 * Returns the intersection between a ranged group and a range.
 * Returns `[]` if the intersection is empty.
 */
export function groupIntersect(range, groups) {
    var e_1, _a;
    var result = [];
    try {
        for (var groups_1 = __values(groups), groups_1_1 = groups_1.next(); !groups_1_1.done; groups_1_1 = groups_1.next()) {
            var r = groups_1_1.value;
            if (range.start >= r.range.end) {
                continue;
            }
            if (range.end < r.range.start) {
                break;
            }
            var intersection = Range.intersect(range, r.range);
            if (Range.isEmpty(intersection)) {
                continue;
            }
            result.push({
                range: intersection,
                size: r.size
            });
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (groups_1_1 && !groups_1_1.done && (_a = groups_1.return)) _a.call(groups_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
    return result;
}
/**
 * Shifts a range by that `much`.
 */
export function shift(_a, much) {
    var start = _a.start, end = _a.end;
    return { start: start + much, end: end + much };
}
/**
 * Consolidates a collection of ranged groups.
 *
 * Consolidation is the process of merging consecutive ranged groups
 * that share the same `size`.
 */
export function consolidate(groups) {
    var e_2, _a;
    var result = [];
    var previousGroup = null;
    try {
        for (var groups_2 = __values(groups), groups_2_1 = groups_2.next(); !groups_2_1.done; groups_2_1 = groups_2.next()) {
            var group = groups_2_1.value;
            var start = group.range.start;
            var end = group.range.end;
            var size = group.size;
            if (previousGroup && size === previousGroup.size) {
                previousGroup.range.end = end;
                continue;
            }
            previousGroup = { range: { start: start, end: end }, size: size };
            result.push(previousGroup);
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (groups_2_1 && !groups_2_1.done && (_a = groups_2.return)) _a.call(groups_2);
        }
        finally { if (e_2) throw e_2.error; }
    }
    return result;
}
/**
 * Concatenates several collections of ranged groups into a single
 * collection.
 */
function concat() {
    var groups = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        groups[_i] = arguments[_i];
    }
    return consolidate(groups.reduce(function (r, g) { return r.concat(g); }, []));
}
var RangeMap = /** @class */ (function () {
    function RangeMap() {
        this.groups = [];
        this._size = 0;
    }
    RangeMap.prototype.splice = function (index, deleteCount, items) {
        if (items === void 0) { items = []; }
        var diff = items.length - deleteCount;
        var before = groupIntersect({ start: 0, end: index }, this.groups);
        var after = groupIntersect({ start: index + deleteCount, end: Number.POSITIVE_INFINITY }, this.groups)
            .map(function (g) { return ({ range: shift(g.range, diff), size: g.size }); });
        var middle = items.map(function (item, i) { return ({
            range: { start: index + i, end: index + i + 1 },
            size: item.size
        }); });
        this.groups = concat(before, middle, after);
        this._size = this.groups.reduce(function (t, g) { return t + (g.size * (g.range.end - g.range.start)); }, 0);
    };
    Object.defineProperty(RangeMap.prototype, "count", {
        /**
         * Returns the number of items in the range map.
         */
        get: function () {
            var len = this.groups.length;
            if (!len) {
                return 0;
            }
            return this.groups[len - 1].range.end;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(RangeMap.prototype, "size", {
        /**
         * Returns the sum of the sizes of all items in the range map.
         */
        get: function () {
            return this._size;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Returns the index of the item at the given position.
     */
    RangeMap.prototype.indexAt = function (position) {
        var e_3, _a;
        if (position < 0) {
            return -1;
        }
        var index = 0;
        var size = 0;
        try {
            for (var _b = __values(this.groups), _c = _b.next(); !_c.done; _c = _b.next()) {
                var group = _c.value;
                var count = group.range.end - group.range.start;
                var newSize = size + (count * group.size);
                if (position < newSize) {
                    return index + Math.floor((position - size) / group.size);
                }
                index += count;
                size = newSize;
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_3) throw e_3.error; }
        }
        return index;
    };
    /**
     * Returns the index of the item right after the item at the
     * index of the given position.
     */
    RangeMap.prototype.indexAfter = function (position) {
        return Math.min(this.indexAt(position) + 1, this.count);
    };
    /**
     * Returns the start position of the item at the given index.
     */
    RangeMap.prototype.positionAt = function (index) {
        var e_4, _a;
        if (index < 0) {
            return -1;
        }
        var position = 0;
        var count = 0;
        try {
            for (var _b = __values(this.groups), _c = _b.next(); !_c.done; _c = _b.next()) {
                var group = _c.value;
                var groupCount = group.range.end - group.range.start;
                var newCount = count + groupCount;
                if (index < newCount) {
                    return position + ((index - count) * group.size);
                }
                position += groupCount * group.size;
                count = newCount;
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_4) throw e_4.error; }
        }
        return -1;
    };
    return RangeMap;
}());
export { RangeMap };
