import {
  BrickConfigVariable,
  BrickCreateUpdateRequest,
  BrickInstance,
} from '@cloud-editor-mono/infrastructure';
import { capitalize } from 'lodash';
import { useEffect, useState } from 'react';

import { Button, ButtonType } from '../../../components-by-app/app-lab';
import { Input } from '../../../essential/input';
import { InputStyle } from '../../../essential/input/input.type';
import { useI18n } from '../../../i18n/useI18n';
import { XSmall, XXSmall, XXXSmall } from '../../../typography';
import { AppLabDialog } from '../app-lab-dialog/AppLabDialog';
import { configureAppBricksDialogMessages as messages } from '../messages';
import styles from './configure-app-bricks-dialog.module.scss';

export type ConfigureAppBricksDialogLogic = () => {
  bricks: BrickInstance[];
  open: boolean;
  confirmAction: (
    bricks: Record<string, BrickCreateUpdateRequest>,
  ) => Promise<boolean>;
  onOpenChange: (open: boolean) => void;
};

type ConfigureAppBricksDialogProps = { logic: ConfigureAppBricksDialogLogic };

interface BrickParam extends Omit<BrickConfigVariable, 'value'> {
  value: string;
}
type BrickParams = Record<string, BrickParam>;
type BricksParams = {
  brick: BrickInstance;
  params: BrickParams;
}[];

export const ConfigureAppBricksDialog: React.FC<
  ConfigureAppBricksDialogProps
> = ({ logic }: ConfigureAppBricksDialogProps) => {
  const { bricks, open, confirmAction, onOpenChange } = logic();
  const [loading, setLoading] = useState(false);
  const [params, setParams] = useState<BricksParams>([]);

  const { formatMessage } = useI18n();

  useEffect(() => {
    if (!open) return;

    setParams(
      bricks.reduce<BricksParams>((acc, brick) => {
        if (
          !brick.id ||
          !brick.config_variables ||
          brick.config_variables.length === 0
        ) {
          return acc;
        }

        return [
          ...acc,
          {
            brick,
            params: brick.config_variables.reduce<BrickParams>(
              (acc, config) => ({
                ...acc,
                [config.name!]: {
                  ...config,
                  value: config.value ?? '',
                },
              }),
              {},
            ),
          },
        ];
      }, []),
    );
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [bricks, open]);

  const handleConfirm = async (): Promise<void> => {
    const variables = params.reduce(
      (acc, { brick, params }) => ({
        ...acc,
        [brick.id!]: {
          variables: Object.keys(params).reduce<Record<string, string>>(
            (acc, key) => ({
              ...acc,
              [key]: params[key].value,
            }),
            {},
          ),
        },
      }),
      {},
    );
    setLoading(true);
    const success = await confirmAction(variables);
    setLoading(false);
    if (success) {
      onOpenChange(false);
    }
  };

  return (
    <AppLabDialog
      open={open}
      onOpenChange={onOpenChange}
      title={formatMessage(messages.dialogTitle)}
      footer={
        <>
          <Button
            type={ButtonType.Secondary}
            onClick={(): void => onOpenChange(false)}
            uppercase={false}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.cancelButton)}
          </Button>
          <Button
            type={ButtonType.Primary}
            uppercase={false}
            loading={loading}
            onClick={handleConfirm}
            disabled={params.some(({ params }) =>
              Object.keys(params).some(
                (key) =>
                  params[key].required && !params[key].value.trim().length,
              ),
            )}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.confirmButton)}
          </Button>
        </>
      }
      classes={{
        root: styles['root'],
        content: styles['content'],
        body: styles['body'],
      }}
    >
      <div className={styles['container']}>
        {params.map(({ brick, params }) => (
          <div key={brick.id} className={styles['brick-container']}>
            <XSmall className={styles['brick-name']}>
              {formatMessage(messages.dialogBodyTitle, {
                brickName: brick.name,
              })}
            </XSmall>
            {Object.keys(params).map((key) => (
              <div key={key} className={styles['param-row']}>
                <Input
                  inputStyle={InputStyle.AppLab}
                  type="text"
                  value={params[key].value}
                  onChange={(value: string): void =>
                    setParams((prev) =>
                      prev.map((it) =>
                        it.brick.id === brick.id
                          ? {
                              ...it,
                              params: {
                                ...it.params,
                                [key]: {
                                  ...it.params[key],
                                  value,
                                },
                              },
                            }
                          : it,
                      ),
                    )
                  }
                  label={params[key].name?.split('_').map(capitalize).join(' ')}
                  required={params[key].required}
                  classes={{
                    input: styles['input'],
                  }}
                />
                <XXSmall>{params[key].description}</XXSmall>
              </div>
            ))}
          </div>
        ))}
        <XXXSmall className={styles['brick-description']}>
          {formatMessage(messages.dialogBodyDescription)}
        </XXXSmall>
      </div>
    </AppLabDialog>
  );
};
