import { Input } from '@bcmi-labs/art-ui';
import {
  ArduinoLogo,
  BoardConnected,
  NoBoard,
  NoDevice,
  Usb,
  Wifi,
} from '@cloud-editor-mono/images/assets/icons';
import { useCallback, useEffect, useState } from 'react';

import { AppLabDialog } from '../../dialogs/app-lab/app-lab-dialog/AppLabDialog';
import {
  AppLabButton as Button,
  ButtonType,
} from '../../essential/app-lab-button';
import { useI18n } from '../../i18n/useI18n';
import { Large, Small } from '../../typography';
import { welcomeMessages } from '../messages';
import { Board } from '../setup.type';
import BoardCard from './BoardCard';
import styles from './welcome.module.scss';

interface WelcomeProps {
  isLoading: boolean;
  boards: Board[];
  onSelectBoard: (board: Board) => void;
  isSelectingBoard?: boolean;
  showBoardConnPswPrompt: boolean;
  onConnPswCancel: () => void;
  onConnPswSubmit: (password: string) => Promise<void>;
  isBoardConnectingOrChecking: boolean;
  connToBoardError?: string;
}

const Welcome: React.FC<WelcomeProps> = (props: WelcomeProps) => {
  const {
    boards,
    onSelectBoard,
    isLoading,
    showBoardConnPswPrompt,
    onConnPswCancel,
    onConnPswSubmit,
    isBoardConnectingOrChecking,
    connToBoardError,
  } = props;

  const { formatMessage } = useI18n();
  const [boardConnPsw, setBoardConnPsw] = useState('');

  const closePasswordPrompt = useCallback((): void => {
    setBoardConnPsw('');
    onConnPswCancel();
  }, [onConnPswCancel]);

  const submitPassword = useCallback((): Promise<void> => {
    return onConnPswSubmit(boardConnPsw);
  }, [boardConnPsw, onConnPswSubmit]);

  const handleKeyDown = useCallback(
    (e: KeyboardEvent): void => {
      switch (e.key) {
        case 'Escape':
          if (!isBoardConnectingOrChecking) {
            closePasswordPrompt();
          }
          break;
        case 'Enter':
          if (boardConnPsw && !isBoardConnectingOrChecking) {
            submitPassword();
          }
          break;
      }
    },
    [
      boardConnPsw,
      closePasswordPrompt,
      isBoardConnectingOrChecking,
      submitPassword,
    ],
  );

  useEffect(() => {
    if (showBoardConnPswPrompt) {
      window.addEventListener('keydown', handleKeyDown);
    }

    return () => {
      window.removeEventListener('keydown', handleKeyDown);
    };
  }, [handleKeyDown, showBoardConnPswPrompt]);

  return (
    <>
      {showBoardConnPswPrompt && (
        <AppLabDialog
          open={showBoardConnPswPrompt}
          title="Linux password"
          closeable={!isBoardConnectingOrChecking}
          classes={{
            root: styles['password-prompt'],
            content: styles['password-prompt-content'],
            body: styles['password-prompt-body'],
            footer: styles['password-prompt-footer'],
          }}
          onOpenChange={(isOpen): void => {
            if (!isOpen) {
              closePasswordPrompt();
            }
          }}
          footer={
            <>
              <Button
                type={ButtonType.Secondary}
                onClick={closePasswordPrompt}
                disabled={isBoardConnectingOrChecking}
              >
                Cancel
              </Button>
              <Button
                type={ButtonType.Primary}
                onClick={submitPassword}
                loading={isBoardConnectingOrChecking}
                disabled={!boardConnPsw || isBoardConnectingOrChecking}
              >
                Confirm
              </Button>
            </>
          }
        >
          <Input type="text" label="Username" value="arduino" disabled />
          <Input
            type="password"
            placeholder="Password"
            error={connToBoardError}
            onChange={(e): void => setBoardConnPsw(e.target.value)}
          />
        </AppLabDialog>
      )}
      <div className={styles['welcome-container']}>
        <ArduinoLogo />
        <div className={styles['welcome-content']}>
          <Large bold className={styles['welcome-title']}>
            {formatMessage(welcomeMessages.title)}
          </Large>
          {isLoading ? null : (
            <>
              <Small bold className={styles['welcome-description']}>
                {formatMessage(welcomeMessages.description)}
              </Small>
              {boards.length > 0 ? (
                <div className={styles['boards']}>
                  {boards.map((board, index) => (
                    <BoardCard
                      key={index} // temp, id is missing
                      title={board.type}
                      chip={board.connectionType}
                      onClick={(): void => onSelectBoard(board)}
                      ChipIcon={
                        board.connectionType === 'Network' ? <Wifi /> : <Usb />
                      }
                      Icon={<BoardConnected />}
                      description={board.name}
                      disabled={board.isSelecting || board.checkingStatus}
                    />
                  ))}
                </div>
              ) : (
                <BoardCard
                  title={formatMessage(welcomeMessages.connectYourBoard)}
                  description={formatMessage(
                    welcomeMessages.connectYourBoardDescription,
                  )}
                  chip={formatMessage(welcomeMessages.noDevice)}
                  ChipIcon={<NoDevice />}
                  Icon={<NoBoard />}
                />
              )}
            </>
          )}
        </div>
      </div>
    </>
  );
};

export default Welcome;
