import clsx from 'clsx';
import { useCallback, useContext, useState } from 'react';

import { Details } from '../../../essential/details';
import { XSmall } from '../../../typography';
import { SidenavContext } from '../../context/sidenavContext';
import { Example, ExamplesFolder, isExamplesFolder } from '../../sidenav.type';
import styles from './examples.module.scss';
import ExamplesList from './ExamplesList';
import ExamplesListLink from './ExamplesListLink';
import { useHighlightText } from './hooks/useHighlightText';

interface ExamplesListItemProps {
  example: Example | ExamplesFolder;
  sourceLibraryID?: string;
  searchQuery?: string;
  index?: number;
  onHeightChange?: (index?: number) => void;
}

const ExamplesListItem: React.FC<ExamplesListItemProps> = (
  props: ExamplesListItemProps,
) => {
  const { example, sourceLibraryID, searchQuery, index, onHeightChange } =
    props;
  const { getCurrentResourceIds } = useContext(SidenavContext);
  const { exampleID } = getCurrentResourceIds();

  const [examplesExpanded, setExamplesExpanded] = useState(false);
  const [selectedExampleExpanded, setSelectedExampleExpanded] = useState(true);

  const isCurrentExampleSelected = useCallback(
    (example: ExamplesFolder): boolean =>
      example.examples.some(
        (exampleItem) =>
          (!isExamplesFolder(exampleItem) && exampleItem.path === exampleID) ||
          (isExamplesFolder(exampleItem) &&
            isCurrentExampleSelected(exampleItem)),
      ),
    [exampleID],
  );

  const toggleExamplesExpanded = useCallback(
    (isOpen: boolean) => {
      onHeightChange && onHeightChange(index);
      !isOpen &&
        isExamplesFolder(example) &&
        isCurrentExampleSelected(example) &&
        setSelectedExampleExpanded(false);
      setExamplesExpanded((prev) => !prev);
    },
    [example, index, isCurrentExampleSelected, onHeightChange],
  );

  const exampleItem = (
    <XSmall title={example.name}>
      {useHighlightText(example.name, searchQuery ?? '')}
    </XSmall>
  );

  return (
    <li
      className={clsx(styles['examples-list-item'], {
        [styles['examples-link-active']]:
          !isExamplesFolder(example) && example.path === exampleID,
      })}
    >
      {isExamplesFolder(example) ? (
        <Details
          summaryNode={exampleItem}
          chevronPosition="left"
          isOpen={
            !!searchQuery ||
            examplesExpanded ||
            (isCurrentExampleSelected(example) && selectedExampleExpanded)
          }
          onToggle={toggleExamplesExpanded}
          classes={{ details: styles['examples-details'] }}
        >
          <ExamplesList
            sourceLibraryID={sourceLibraryID}
            items={example.examples}
            searchQuery={searchQuery}
          />
        </Details>
      ) : (
        <ExamplesListLink
          example={example}
          sourceLibraryID={sourceLibraryID}
          searchQuery={searchQuery}
        />
      )}
    </li>
  );
};

export default ExamplesListItem;
