import React, { ForwardedRef, forwardRef, LegacyRef, useCallback } from 'react';
import {
  AutoSizer,
  CellMeasurer,
  CellMeasurerCache,
  Index,
  InfiniteLoader,
  List,
  ListRowProps,
} from 'react-virtualized';

import { GetLibrary, SidenavStandardLibrary } from '../../sidenav.type';
import NoResults from '../common/NoResults';
import styles from './examples.module.scss';
import ExamplesFromLibrariesListItem from './ExamplesFromLibrariesListItem';
import ExamplesListItemSkeleton from './ExamplesListItemSkeleton';

interface ExamplesFromLibrariesVirtualListProps {
  libraries: SidenavStandardLibrary[];
  filter?: (lib: SidenavStandardLibrary) => boolean;
  getLibraryDetails: (item: SidenavStandardLibrary) => ReturnType<GetLibrary>;
  isLoading?: boolean;
  cache: CellMeasurerCache;
  getMoreItems: () => Promise<unknown>;
  isGettingMoreItems: boolean;
  hasMoreItems?: boolean;
}

const ExamplesFromLibrariesVirtualList = forwardRef(
  (
    props: ExamplesFromLibrariesVirtualListProps,
    listRef: ForwardedRef<List | undefined>,
  ) => {
    const {
      libraries,
      filter,
      getLibraryDetails,
      isLoading,
      cache,
      getMoreItems,
      isGettingMoreItems,
      hasMoreItems,
    } = props;

    const rowCount = !libraries
      ? 0
      : hasMoreItems
      ? libraries.length + 1
      : libraries.length;

    const loadMoreRows = isGettingMoreItems
      ? async (): Promise<void> => {
          return;
        }
      : getMoreItems;

    const isRowLoaded = ({ index }: Index): boolean =>
      !hasMoreItems || Boolean(libraries && index < libraries.length);

    const onHeightChange = useCallback(
      (index?: number) => {
        if (typeof index !== 'undefined') {
          cache.clear(index, 0);
          listRef &&
            (
              listRef as React.MutableRefObject<List | undefined>
            ).current?.recomputeRowHeights(index);
        }
      },
      [cache, listRef],
    );

    const rowRenderer = ({
      key,
      index,
      parent,
      style,
    }: ListRowProps): React.ReactNode => {
      return (
        <CellMeasurer
          key={key}
          cache={cache}
          parent={parent}
          columnIndex={0}
          rowIndex={index}
        >
          {({ registerChild }): React.ReactNode => {
            if (!isRowLoaded({ index })) {
              return (
                <ExamplesListItemSkeleton
                  ref={registerChild}
                  key={`examples-from-libs-list-item-skeleton-${key}`}
                  style={style}
                />
              );
            }

            const library = (libraries &&
              libraries[index]) as SidenavStandardLibrary;

            if (library.examplesNumber < 1 || (filter && !filter(library))) {
              return null;
            }

            return (
              <ExamplesFromLibrariesListItem
                ref={registerChild}
                libraryItem={library}
                getLibraryDetails={getLibraryDetails}
                style={style}
                index={index}
                onHeightChange={onHeightChange}
              />
            );
          }}
        </CellMeasurer>
      );
    };

    const skeletonChildren = Number(styles.skeletonChildren);

    return (
      <>
        {!isLoading && (!libraries || libraries.length === 0) ? (
          <NoResults
            resourceName="examples"
            classes={{ container: styles['no-results-container'] }}
          />
        ) : (
          <div className={styles['examples-virtual-list-container']}>
            {isLoading ? (
              <ExamplesListItemSkeleton count={skeletonChildren} />
            ) : (
              <InfiniteLoader
                isRowLoaded={isRowLoaded}
                loadMoreRows={loadMoreRows}
                rowCount={rowCount}
              >
                {({ onRowsRendered }): React.ReactNode => (
                  <AutoSizer>
                    {({ width, height }): React.ReactNode => (
                      <List
                        ref={listRef as LegacyRef<List>}
                        width={width}
                        height={height}
                        rowCount={rowCount}
                        rowHeight={cache.rowHeight}
                        rowRenderer={rowRenderer}
                        onRowsRendered={onRowsRendered}
                        deferredMeasurementCache={cache}
                      />
                    )}
                  </AutoSizer>
                )}
              </InfiniteLoader>
            )}
          </div>
        )}
      </>
    );
  },
);

ExamplesFromLibrariesVirtualList.displayName =
  'ExamplesFromLibrariesVirtualList';
export default ExamplesFromLibrariesVirtualList;
