import { FileNode, FolderNode, TreeNode } from './fileTree.type';

export const isFileNode = (node: TreeNode): node is FileNode => {
  return node.type === 'file';
};

export const isFolderNode = (node: TreeNode): node is FolderNode => {
  return node.type === 'folder';
};

export const countAllNodes = (nodes?: TreeNode[]): number => {
  if (!nodes) {
    return 0;
  }

  return nodes.reduce((total, node) => total + countNodes(node), 0);
};

export const countNodes = (node?: TreeNode): number => {
  if (!node) {
    return 0;
  }

  if (node.type === 'file' || !node.children) {
    return 1;
  }

  return 1 + node.children.reduce((sum, child) => sum + countNodes(child), 0);
};

export function insertNewNode(tree: TreeNode[], path: string): TreeNode[] {
  const parts = path ? path.split('/') : [];
  const clone = JSON.parse(JSON.stringify(tree)) as TreeNode[];
  let current: TreeNode[] = clone;

  for (const part of parts) {
    if (!part) {
      break;
    }
    const folder = current.find((n) => n.name === part);
    if (!folder || isFileNode(folder)) {
      break;
    }
    if (isFolderNode(folder)) {
      if (!folder.children) {
        folder.children = [];
      }
      current = folder.children;
    }
  }

  current.push({
    path: `${path}/__CREATE__`,
    name: '',
    type: 'file',
    mimeType: '',
    extension: '',
  });

  return clone;
}

export const formatBytes = (
  bytes: number | undefined,
  decimals = 1,
): string => {
  if (bytes === undefined || bytes < 0) return 'N/A';
  if (bytes === 0) return '0 bytes';
  if (bytes < 1024) return `${bytes} byte${bytes === 1 ? '' : 's'}`;

  const k = 1024;
  const sizes = ['KB', 'MB', 'GB', 'TB', 'PB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  const value = bytes / Math.pow(k, i);

  return `${parseFloat(value.toFixed(decimals))} ${sizes[i - 1]}`;
};

export const formatDate = (dateStr: string): string => {
  const date = new Date(dateStr);
  const now = new Date();

  const isToday = date.toDateString() === now.toDateString();

  const yesterday = new Date();
  yesterday.setDate(now.getDate() - 1);
  const isYesterday = date.toDateString() === yesterday.toDateString();

  const time = date.toLocaleTimeString([], {
    hour: '2-digit',
    minute: '2-digit',
  });

  if (isToday) return `Today, ${time}`;
  if (isYesterday) return `Yesterday, ${time}`;

  const day = date.getDate();
  const month = date.toLocaleString('en-US', { month: 'short' });
  const year = date.getFullYear();

  return `${day} ${month} ${year}, ${time}`;
};
