import * as ContextMenu from '@radix-ui/react-context-menu';
import clsx from 'clsx';
import { RowRendererProps } from 'react-arborist';

import styles from './file-tree.module.scss';
import { TreeNode } from './fileTree.type';
import { isFileNode } from './utils';

type FileRowProps = RowRendererProps<TreeNode> & {
  selectedNode: TreeNode | undefined;
  onSelect: () => void;
  onRename: () => void;
  onDelete: () => void;
  isReadOnly: boolean;
  isBricksSelected?: boolean;
};

const FileRow: React.FC<FileRowProps> = ({
  selectedNode,
  onSelect,
  onRename,
  onDelete,
  isReadOnly,
  isBricksSelected = false,
  ...rowProps
}: FileRowProps) => {
  const { node } = rowProps;

  return (
    <div {...rowProps.attrs} className={styles['tree-row-container']}>
      <ContextMenu.Root>
        <ContextMenu.Trigger
          disabled={isReadOnly || !isFileNode(node.data)}
          className={styles['tree-row-context-menu-trigger']}
        >
          <div
            role="button"
            tabIndex={0}
            className={clsx(styles['tree-row'], {
              [styles['tree-row-selected']]:
                isFileNode(node.data) &&
                selectedNode &&
                node.data.path === selectedNode.path &&
                !isBricksSelected,
            })}
            ref={rowProps.innerRef}
            onFocus={(e): void => e.stopPropagation()}
            onKeyDown={(e): void => {
              if (e.key === 'Enter') {
                onSelect();
              }
            }}
            onClick={(): void => onSelect()}
          >
            {rowProps.children}
          </div>
        </ContextMenu.Trigger>

        {isFileNode(node.data) && (
          <ContextMenu.Portal>
            <ContextMenu.Content className={styles['tree-row-context-menu']}>
              <ContextMenu.Item
                className={styles['tree-row-context-menu-item']}
                onSelect={onRename}
              >
                Rename
              </ContextMenu.Item>
              <ContextMenu.Item
                className={styles['tree-row-context-menu-item']}
                onSelect={onDelete}
              >
                Delete
              </ContextMenu.Item>
            </ContextMenu.Content>
          </ContextMenu.Portal>
        )}
      </ContextMenu.Root>
    </div>
  );
};

export default FileRow;
