import {
  CaretDown as CaretDownIcon,
  FileConfig as FileConfigIcon,
  FileFolder as FileFolderIcon,
  FileGeneric as FileGenericIcon,
  FileIno as FileInoIcon,
  FilePython as FilePythonIcon,
  FileText as FileTextIcon,
} from '@cloud-editor-mono/images/assets/icons';
import clsx from 'clsx';
import { useCallback, useEffect, useRef, useState } from 'react';
import { NodeRendererProps } from 'react-arborist';

import styles from './file-tree.module.scss';
import { TreeNode } from './fileTree.type';
import { isFolderNode } from './utils';

type FileNodeProps = NodeRendererProps<TreeNode> & {
  isEditing: boolean;
  isReadOnly: boolean;
  onEditStart: () => void;
  onEditSubmit: (newName: string) => Promise<void>;
  onEditCancel: () => void;
  onDelete: () => Promise<void>;
};

const renderFileIcon = (node: TreeNode): JSX.Element => {
  if (isFolderNode(node)) {
    return <FileFolderIcon />;
  }

  if (node.name === 'app.yaml') {
    return <FileConfigIcon />;
  }
  switch (node.extension) {
    case '.py':
      return <FilePythonIcon />;
    case '.ino':
      return <FileInoIcon />;
    case '.txt':
      return <FileTextIcon />;
    case '.json':
    default:
      return <FileGenericIcon />;
  }
};

const FileNode: React.FC<FileNodeProps> = ({
  node,
  style,
  dragHandle,
  isEditing,
  isReadOnly,
  onEditStart,
  onEditSubmit,
  onEditCancel,
}: FileNodeProps) => {
  const inputRef = useRef<HTMLInputElement>(null);

  const [value, setValue] = useState<string>(node.data.name);

  const preSubmit = useCallback((): void => {
    if (value && value !== node.data.name) {
      onEditSubmit(value);
    } else {
      onEditCancel();
    }
  }, [node.data.name, onEditCancel, onEditSubmit, value]);

  useEffect(() => {
    if (isEditing && inputRef.current) {
      inputRef.current.focus();
      inputRef.current.select();
    }
  }, [isEditing]);

  return (
    <div
      ref={dragHandle}
      style={style}
      className={clsx(styles.node, node.state, styles['tree-node'])}
      onDoubleClick={async (e): Promise<void> => {
        if (isReadOnly) {
          return;
        }
        if (isFolderNode(node.data)) {
          // Temporary disable folder rename
          return;
        }
        e.stopPropagation();
        onEditStart();
      }}
    >
      {node.isInternal && (
        <CaretDownIcon
          className={node.isClosed ? styles['tree-node-closed'] : ''}
        />
      )}

      <div className={styles['tree-node-icon']}>
        {renderFileIcon(node.data)}
      </div>

      {!isEditing && node.data.name}

      {isEditing && (
        <input
          ref={inputRef}
          className={styles['tree-node-input']}
          autoCorrect="off"
          autoCapitalize="off"
          spellCheck="false"
          value={value}
          onChange={(e): void => setValue(e.target.value)}
          onBlur={(): void => {
            preSubmit();
          }}
          onClick={(e): void => e.stopPropagation()}
          onKeyDown={(e): void => {
            e.stopPropagation();

            if (e.key === 'Enter') {
              preSubmit();
            } else if (e.key === 'Escape') {
              onEditCancel();
            }
          }}
        />
      )}
    </div>
  );
};

export default FileNode;
