import {
  BrickCreateUpdateRequest,
  BrickDetails,
  BrickInstance,
  BrickVariable,
} from '@cloud-editor-mono/infrastructure';
import { useEffect, useState } from 'react';

import { Button, ButtonType } from '../../../components-by-app/app-lab';
import { Input } from '../../../essential/input';
import { InputStyle } from '../../../essential/input/input.type';
import { useI18n } from '../../../i18n/useI18n';
import { Medium } from '../../../typography';
import { AppLabDialog } from '../app-lab-dialog/AppLabDialog';
import { configureAppBrickDialogMessages as messages } from '../messages';
import styles from './configure-app-brick-dialog.module.scss';

export type ConfigureAppBrickDialogLogic = () => {
  brick: BrickDetails;
  open: boolean;
  loadBrickInstance: (id: string) => Promise<BrickInstance>;
  confirmAction: (
    brickId: string,
    params: BrickCreateUpdateRequest,
  ) => Promise<boolean>;
  onOpenChange: (open: boolean) => void;
};

type ConfigureAppBrickDialogProps = { logic: ConfigureAppBrickDialogLogic };

interface BrickParam extends BrickVariable {
  value: string;
}
type BrickParams = Record<string, BrickParam>;

export const ConfigureAppBrickDialog: React.FC<
  ConfigureAppBrickDialogProps
> = ({ logic }: ConfigureAppBrickDialogProps) => {
  const { brick, open, loadBrickInstance, confirmAction, onOpenChange } =
    logic();
  const [loading, setLoading] = useState(false);
  const [params, setParams] = useState<BrickParams>({});

  const { formatMessage } = useI18n();

  useEffect(() => {
    const loadInstance = async (): Promise<void> => {
      if (!brick.id) return;
      try {
        const instance = await loadBrickInstance(brick.id);
        setParams(
          Object.keys(brick.variables ?? {}).reduce(
            (acc, key) => ({
              ...acc,
              [key]: {
                ...brick.variables?.[key],
                value: instance.variables?.[key] ?? '',
              },
            }),
            {},
          ),
        );
      } catch {
        setParams({});
      }
    };
    loadInstance();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [brick]);

  const handleConfirm = async (): Promise<void> => {
    if (!brick.id) return;
    const variables = Object.keys(params).reduce(
      (acc, key) => ({
        ...acc,
        [key]: params[key].value,
      }),
      {},
    );
    setLoading(true);
    const success = await confirmAction(brick.id, {
      variables,
    });
    setLoading(false);
    if (success) {
      onOpenChange(false);
    }
  };

  return (
    <AppLabDialog
      open={open}
      onOpenChange={onOpenChange}
      title={formatMessage(messages.dialogTitle)}
      footer={
        <>
          <Button
            type={ButtonType.Secondary}
            onClick={(): void => onOpenChange(false)}
            uppercase={false}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.cancelButton)}
          </Button>
          <Button
            type={ButtonType.Primary}
            uppercase={false}
            loading={loading}
            onClick={handleConfirm}
            disabled={Object.keys(params).some(
              (key) => params[key].required && !params[key].value.trim().length,
            )}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.confirmButton)}
          </Button>
        </>
      }
      classes={{
        root: styles['root'],
        content: styles['content'],
        body: styles['body'],
      }}
    >
      <div className={styles['container']}>
        <Medium className={styles['brick-name']}>{brick.name}</Medium>
        {Object.keys(params).map((key) => (
          <div key={key} className={styles['param-row']}>
            <Input
              inputStyle={InputStyle.AppLab}
              type="text"
              value={params[key].value}
              onChange={(value: string): void =>
                setParams((prev) => ({
                  ...prev,
                  [key]: {
                    ...prev[key],
                    value,
                  },
                }))
              }
              label={params[key].description}
              required={params[key].required}
              classes={{
                input: styles['input'],
              }}
            />
          </div>
        ))}
      </div>
    </AppLabDialog>
  );
};
