import {
  BrickInstance,
  BrickListItem,
} from '@cloud-editor-mono/infrastructure';
import clsx from 'clsx';
import { useEffect, useState } from 'react';
import Split from 'react-split';

import { BrickDetailLogic } from '../../../app-lab-brick-detail';
import BrickDetail from '../../../app-lab-brick-detail/BrickDetail';
import { AppLabBricksList } from '../../../app-lab-bricks-list';
import { Button, ButtonType } from '../../../components-by-app/app-lab';
import { useI18n } from '../../../i18n/useI18n';
import { AppLabDialog } from '../app-lab-dialog/AppLabDialog';
import { addAppBrickDialogMessages as messages } from '../messages';
import styles from './add-app-brick-dialog.module.scss';

export type AddAppBrickDialogLogic = () => {
  appBricks: BrickInstance[] | undefined;
  bricks: BrickListItem[];
  open: boolean;
  brickDetailLogic: BrickDetailLogic;
  confirmAction: (brick: BrickListItem) => Promise<boolean>;
  onOpenChange: (open: boolean) => void;
};

type AddAppBrickDialogProps = { logic: AddAppBrickDialogLogic };

export const AddAppBrickDialog: React.FC<AddAppBrickDialogProps> = ({
  logic,
}: AddAppBrickDialogProps) => {
  const {
    appBricks,
    bricks,
    open,
    brickDetailLogic,
    confirmAction,
    onOpenChange,
  } = logic();
  const [loading, setLoading] = useState(false);
  const [selectedBrick, setSelectedBrick] = useState<BrickListItem>();

  useEffect(() => {
    setSelectedBrick(
      bricks.find((brick) => appBricks?.every((it) => it.id !== brick.id)),
    );
  }, [appBricks, bricks, open]);

  const { formatMessage } = useI18n();

  const handleConfirm = async (): Promise<void> => {
    if (!selectedBrick) return;
    setLoading(true);
    const success = await confirmAction(selectedBrick);
    setLoading(false);
    if (success) {
      onOpenChange(false);
    }
  };

  return (
    <AppLabDialog
      open={open}
      onOpenChange={onOpenChange}
      title={formatMessage(messages.dialogTitle)}
      footer={
        <>
          <Button
            type={ButtonType.Secondary}
            onClick={(): void => onOpenChange(false)}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.cancelButton)}
          </Button>
          <Button
            type={ButtonType.Primary}
            loading={loading}
            disabled={appBricks?.some(
              (appBrick) => appBrick.id === selectedBrick?.id,
            )}
            onClick={handleConfirm}
            classes={{
              button: styles['action-button'],
              textButtonText: styles['action-button-text'],
            }}
          >
            {formatMessage(messages.confirmButton)}
          </Button>
        </>
      }
      classes={{
        root: styles['root'],
        content: styles['content'],
        body: styles['body'],
      }}
    >
      <Split
        className={styles['split']}
        sizes={[30, 70]}
        minSize={400}
        expandToMin={false}
        gutterSize={26}
        gutterAlign="center"
        snapOffset={30}
        direction="horizontal"
        cursor="col-resize"
        gutter={(): HTMLElement => {
          const element = document.createElement('div');
          element.className = styles['gutter'];
          return element;
        }}
      >
        <div className={(styles['split-item'], styles['split-item-left'])}>
          <AppLabBricksList
            bricks={bricks}
            disabledBricks={bricks.filter((brick) =>
              appBricks?.some((appBrick) => appBrick.id === brick.id),
            )}
            selectedBrick={selectedBrick}
            brickSize="medium"
            expanded={false}
            onClick={setSelectedBrick}
            classes={{
              container: styles['bricks-list-container'],
              item: styles['brick-item'],
              itemSelected: styles['selected'],
              itemTitle: styles['brick-item-title'],
            }}
          />
        </div>

        <div className={clsx(styles['split-item'], styles['split-item-right'])}>
          <BrickDetail
            brickId={selectedBrick?.id ?? ''}
            brickDetailLogic={brickDetailLogic}
          />
        </div>
      </Split>
    </AppLabDialog>
  );
};
