import { SketchLibrary } from '@cloud-editor-mono/infrastructure';
import React, { useCallback, useState } from 'react';

import { Button, ButtonSize, ButtonType } from '../components-by-app/app-lab';
import { Text, TextSize } from '../typography';
import styles from './sketch-library-card.module.scss';

interface SketchLibraryCardProps {
  library: SketchLibrary;
  onInstall: (libRef: string) => void;
  isInstalling?: boolean;
  installedPackage: string | undefined;
  onDelete: (libRef: string) => void;
  isDeleting?: boolean;
}

const SketchLibraryCard = (
  props: SketchLibraryCardProps,
): React.ReactElement => {
  const {
    library,
    onInstall,
    isInstalling,
    installedPackage,
    onDelete,
    isDeleting,
  } = props;

  const [selectedVersion, setSelectedVersion] = useState<string | undefined>(
    undefined,
  );

  const onClick = useCallback((): void => {
    if (isInstalling || isDeleting) {
      return;
    }

    if (installedPackage) {
      return onDelete(`${library.id}@${installedPackage}`);
    }

    const libRef = selectedVersion
      ? `${library.id}@${selectedVersion}`
      : library.id;
    libRef && onInstall(libRef);
  }, [
    installedPackage,
    isDeleting,
    isInstalling,
    library.id,
    onDelete,
    onInstall,
    selectedVersion,
  ]);

  return (
    <div className={styles['lib-container']}>
      <div className={styles['lib-header']}>
        <Text bold size={TextSize.XSmall}>
          {library.name}
        </Text>
        <Text size={TextSize.XXXSmall}>{library.author}</Text>
      </div>
      <p className={styles['lib-sentence']}>{library.sentence}</p>
      <div className={styles['lib-actions']}>
        <select
          value={installedPackage}
          disabled={!!installedPackage}
          onChange={(e): void => {
            setSelectedVersion(e.target.value);
          }}
          className={styles['lib-select']}
        >
          {library.releases?.map((r) => (
            <option key={r.id} value={r.version}>
              {r.version || ''}
            </option>
          ))}
        </select>
        <Button
          onClick={onClick}
          type={installedPackage ? ButtonType.Secondary : ButtonType.Primary}
          size={ButtonSize.XSmall}
          loading={isInstalling || isDeleting}
        >
          {installedPackage ? 'Remove' : 'Install'}
        </Button>
      </div>
    </div>
  );
};

export default SketchLibraryCard;
