import {
  Duplicate,
  Play,
  Spinner,
  StatusError,
  StatusSuccess,
  Stop,
} from '@cloud-editor-mono/images/assets/icons';
import {
  Button,
  ButtonSize,
  ButtonType,
  ButtonVariant,
} from '@cloud-editor-mono/ui-components/lib/components-by-app/app-lab';
import clsx from 'clsx';
import { useEffect, useRef, useState } from 'react';

import styles from './runtime-actions.module.scss';
import {
  AppLabAction,
  AppLabActionStatus,
  RuntimeActionsProps,
} from './runtimeActions.type';

const RuntimeActions = <T extends string>(
  props: RuntimeActionsProps<T>,
): React.ReactElement => {
  const {
    runtimeActionsLogic,
    setTab,
    runtimeDisable,
    size = 'default',
  } = props;

  const {
    appId,
    appName,
    appStatus,
    currentAction,
    currentActionStatus,
    duplicateApp,
    runApp,
    stopApp,
    isBannerEnabled = true,
    showStop = true,
  } = runtimeActionsLogic();

  const stopHandler = (): void => {
    if (appId) {
      stopApp(appId, appStatus);
      setTab && setTab('console' as T);
    }
  };

  const runHandler = (): void => {
    if (appId) {
      runApp(appId);
      setTab && setTab('console' as T);
    }
  };

  const getStatusBanner = (params: {
    action: AppLabAction;
    actionStatus: AppLabActionStatus;
  }): JSX.Element => {
    const messageDictionary: Record<
      AppLabActionStatus,
      Record<AppLabAction, string>
    > = {
      [AppLabActionStatus.Idle]: {
        run: '',
        stop: '',
        logs: '',
      },
      [AppLabActionStatus.Pending]: {
        run: 'Running...',
        stop: 'Stopping...',
        logs: '',
      },
      [AppLabActionStatus.Succeeded]: {
        run: `Done ${appName}.yml`,
        stop: `Stopped ${appName}.yml`,
        logs: `Done ${appName}.yml`,
      },
      [AppLabActionStatus.Errored]: {
        run: `Failed ${appName}.yml`,
        stop: `Failed ${appName}.yml`,
        logs: '',
      },
    };

    const iconDictionary: Record<AppLabActionStatus, JSX.Element> = {
      [AppLabActionStatus.Idle]: <></>,
      [AppLabActionStatus.Pending]: <Spinner />,
      [AppLabActionStatus.Succeeded]: <StatusSuccess />,
      [AppLabActionStatus.Errored]: <StatusError />,
    };

    const { action, actionStatus } = params;
    const message = messageDictionary[actionStatus]?.[action] || '';
    const icon = iconDictionary[actionStatus] || <></>;

    return (
      <div
        className={clsx(styles['status-message'], styles[actionStatus], {
          [styles['hide-after-3s']]:
            actionStatus !== AppLabActionStatus.Pending,
        })}
      >
        {icon}
        <span>{message}</span>
      </div>
    );
  };

  const canAbortRun =
    currentActionStatus === AppLabActionStatus.Pending &&
    currentAction === AppLabAction.Run &&
    appStatus === 'stopped';

  const stopShown = showStop && (appStatus === 'running' || canAbortRun);
  const stopDisabled =
    currentActionStatus === AppLabActionStatus.Pending &&
    currentAction === AppLabAction.Stop;

  const BANNER_CLEAN_TIME = 3500; //Animation is on 3s lasting 500ms
  const [showStatusBanner, setShowStatusBanner] = useState<
    | {
        action: AppLabAction;
        actionStatus: AppLabActionStatus;
      }
    | undefined
  >();

  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  useEffect(() => {
    clearTimeout(timeoutRef.current || undefined);

    if (!currentAction || !isBannerEnabled)
      return setShowStatusBanner(undefined);

    setShowStatusBanner({
      action: currentAction,
      actionStatus: currentActionStatus,
    });

    if (currentActionStatus !== AppLabActionStatus.Pending) {
      timeoutRef.current = setTimeout(() => {
        setShowStatusBanner(undefined);
        timeoutRef.current = null;
      }, BANNER_CLEAN_TIME);
    }

    return () => clearTimeout(timeoutRef.current || undefined);
  }, [currentAction, currentActionStatus, isBannerEnabled]);

  return (
    <div className={styles['actions']}>
      {showStatusBanner && getStatusBanner(showStatusBanner)}
      {stopShown && (
        <Button
          onClick={stopHandler}
          type={ButtonType.Secondary}
          size={size === 'small' ? ButtonSize.XSmall : ButtonSize.Small}
          Icon={Stop}
          variant={ButtonVariant.Destructive}
          disabled={stopDisabled || runtimeDisable}
        >
          Stop
        </Button>
      )}
      {!stopShown && (
        <Button
          onClick={runHandler}
          type={ButtonType.Primary}
          size={size === 'small' ? ButtonSize.XSmall : ButtonSize.Small}
          Icon={Play}
          variant={ButtonVariant.Action}
          disabled={
            currentActionStatus === AppLabActionStatus.Pending || runtimeDisable
          }
        >
          Run
        </Button>
      )}
      {duplicateApp && (
        <Button
          onClick={duplicateApp}
          type={ButtonType.Secondary}
          size={ButtonSize.Small}
          Icon={Duplicate}
          variant={ButtonVariant.Action}
          disabled={runtimeDisable}
          classes={{
            button: styles['actions--duplicate'],
          }}
        >
          Copy and edit app
        </Button>
      )}
    </div>
  );
};

export default RuntimeActions;
