import { Bin } from '@cloud-editor-mono/images/assets/icons';
import clsx from 'clsx';
import { MouseEvent as ReactMouseEvent, useEffect, useState } from 'react';

import BrickIcon from '../app-lab-brick-icon/BrickIcon';
import { XXSmall } from '../typography';
import styles from './brick-item.module.scss';
import { BrickItemProps } from './BrickItem.type';

const BrickItem: React.FC<BrickItemProps> = (props: BrickItemProps) => {
  const { brick, selected, onClick, onDelete } = props;
  const [open, setOpen] = useState(false);

  useEffect(() => {
    const handleClickOutside = (e: MouseEvent): void => {
      if (
        !(e.target as HTMLElement).closest(
          `.${styles['brick-item-context-menu']}`,
        )
      ) {
        setOpen(false);
      }
    };

    window.addEventListener('mousedown', handleClickOutside);
    return () => {
      window.removeEventListener('mousedown', handleClickOutside);
    };
  }, []);

  const handleContextMenu = onDelete
    ? (e: ReactMouseEvent): void => {
        e.preventDefault();
        setOpen(true);
      }
    : undefined;

  const handleDelete = (e: ReactMouseEvent): void => {
    e.stopPropagation();
    setOpen(false);
    onDelete?.();
  };

  return (
    <div
      role="button"
      tabIndex={0}
      className={clsx(styles['brick-item'], {
        [styles['active']]: open || selected,
      })}
      onClick={onClick}
      onContextMenu={handleContextMenu}
      onKeyDown={(e): void => {
        if (e.key === 'Enter') {
          onClick?.();
        }
      }}
    >
      <BrickIcon category={brick.category} size="xsmall" />
      <XXSmall className={styles['brick-item-name']}>{brick.name}</XXSmall>
      {open && (
        <div className={styles['brick-item-context-menu']}>
          <button
            className={styles['brick-item-delete-button']}
            onClick={handleDelete}
          >
            <Bin />
            <XXSmall>Remove</XXSmall>
          </button>
        </div>
      )}
    </div>
  );
};

export default BrickItem;
