import { Badge } from '@bcmi-labs/art-ui';
import {
  Bin,
  CaretDown,
  Duplicate,
  Pencil,
  PlayOutline,
  Spinner,
} from '@cloud-editor-mono/images/assets/icons';
import clsx from 'clsx';
import { useRef, useState } from 'react';

import { AppLabEmojiPicker } from '../app-lab-emoji-picker';
import { CreateAppDialog, DeleteAppDialog } from '../dialogs';
import { SetAsDefaultAppDialog } from '../dialogs/app-lab/set-as-default-app-dialog/SetAsDefaultAppDialog';
import { DropdownMenuButton } from '../essential/dropdown-menu/DropdownMenuButton';
import { Input } from '../essential/input';
import { InputStyle } from '../essential/input/input.type';
import { useI18n } from '../i18n/useI18n';
import { XSmall } from '../typography';
import styles from './app-title.module.scss';
import { AppAction, AppTitleLogic } from './AppTitle.type';
import { appTitleMessages } from './messages';

interface AppTitleProps {
  appTitleLogic: AppTitleLogic;
  showBadge?: boolean;
  useStaticPosition?: boolean;
  loader?: boolean;
  type?: 'default' | 'footer';
  disabled?: boolean;
}

const AppTitle: React.FC<AppTitleProps> = (props: AppTitleProps) => {
  const {
    appTitleLogic,
    showBadge = true,
    useStaticPosition,
    loader,
    type,
    disabled,
  } = props;
  const inputRef = useRef<HTMLDivElement>(null);
  const [open, setOpen] = useState(false);
  const {
    app,
    deleteAppDialogLogic,
    createAppDialogLogic,
    setAsDefaultAppDialogLogic,
    name,
    editing,
    hasError,
    onAppNameChange,
    onAppAction,
    onResetAppName,
    onRenameApp,
    onUpdateAppIcon,
  } = appTitleLogic();
  const { formatMessage } = useI18n();

  return (
    <div
      className={clsx(styles['app-title'], {
        [styles['active']]: editing || open,
        [styles['example']]: app?.example,
        [styles['default']]: type === 'default',
        [styles['footer']]: type === 'footer',
        [styles['disabled']]: !!disabled,
      })}
    >
      <DeleteAppDialog logic={deleteAppDialogLogic} />
      <CreateAppDialog logic={createAppDialogLogic} />
      <SetAsDefaultAppDialog logic={setAsDefaultAppDialogLogic} />
      <div className={styles['app-icon']}>
        {app?.example ? (
          app.icon
        ) : (
          <AppLabEmojiPicker
            value={app?.icon}
            onChange={onUpdateAppIcon}
            onOpen={setOpen}
          />
        )}
      </div>
      <div className={styles['app-name']}>
        {editing ? (
          <Input
            ref={inputRef}
            inputStyle={InputStyle.AppLab}
            // eslint-disable-next-line jsx-a11y/no-autofocus
            autoFocus
            type="text"
            value={name}
            blurOnEnter={false}
            error={
              hasError
                ? new Error(formatMessage(appTitleMessages.appNameInUse))
                : undefined
            }
            onChange={onAppNameChange}
            onBlur={onResetAppName}
            onEnter={onRenameApp}
            onKeyDown={(e): void => {
              if (e.key === 'Escape') {
                onResetAppName();
              }
            }}
            classes={{
              input: styles['app-name-input'],
              inputError: styles['error-message'],
              error: styles['app-name-input-error'],
            }}
            styles={{
              inputError: {
                left: inputRef.current?.getBoundingClientRect().left,
              },
            }}
          />
        ) : (
          <div
            {...(!app?.example && {
              onClick: (): void => onAppAction(AppAction.Rename),
              onKeyUp: (): void => onAppAction(AppAction.Rename),
            })}
          >
            <XSmall className={styles['app-name-text']}>{app?.name}</XSmall>
          </div>
        )}
      </div>
      {(!app?.example || type === 'footer') && (
        <DropdownMenuButton
          disabled={disabled}
          sections={[
            {
              name: 'Actions',
              items: [
                {
                  id: AppAction.SetAsDefault,
                  label: formatMessage(
                    app?.default
                      ? appTitleMessages.actionRemoveAsDefault
                      : appTitleMessages.actionSetAsDefault,
                  ),
                  labelPrefix: <PlayOutline />,
                },
                ...(!app?.example
                  ? [
                      {
                        id: AppAction.Rename,
                        label: formatMessage(appTitleMessages.actionRename),
                        labelPrefix: <Pencil />,
                      },
                    ]
                  : []),
                {
                  id: AppAction.Duplicate,
                  label: formatMessage(appTitleMessages.actionDuplicate),
                  labelPrefix: <Duplicate />,
                },
                ...(!app?.example
                  ? [
                      {
                        id: AppAction.Delete,
                        itemClassName: styles['danger'],
                        label: formatMessage(appTitleMessages.actionDelete),
                        labelPrefix: <Bin />,
                      },
                    ]
                  : []),
              ],
            },
          ]}
          classes={{
            dropdownMenu: styles['dropdown-menu'],
            dropdownMenuButton: clsx(styles['dropdown-menu-button'], {
              [styles['loading']]: loader,
            }),
            dropdownMenuButtonOpen: styles['dropdown-menu-button-open'],
            dropdownMenuButtonWrapper: clsx(styles['app-actions'], {
              //TODO: Remove this once the dropdown menu is set in the footer
              [styles['dropdown-menu-button-footer']]:
                type === 'footer' && !loader,
            }),
            dropdownMenuItem: styles['dropdown-menu-item'],
          }}
          onAction={(key): void => onAppAction(key as AppAction)}
          onOpen={setOpen}
          buttonChildren={loader ? <Spinner /> : <CaretDown />}
          useStaticPosition={useStaticPosition}
        />
      )}
      {app?.default && showBadge && (
        <Badge className={styles['default-badge']}>
          {formatMessage(appTitleMessages.appDefault)}
        </Badge>
      )}
    </div>
  );
};

export default AppTitle;
