package network

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/bcmi-labs/orchestrator/pkg/board/remote"
)

type Status string

var (
	ConnectedStatus    Status = "connected"
	ConnectingStatus   Status = "connecting"
	DisconnectedStatus Status = "disconnected"
)

func (m *Manager) GetStatusByType(ctx context.Context, netType string) (Status, error) {
	// -t = terse, -f = columns  TYPE,STATE
	out, err := m.Run(ctx, "-t", "-f", "TYPE,STATE", "device")
	if err != nil {
		return DisconnectedStatus, fmt.Errorf("failed to query devices: %w", err)
	}

	for _, line := range strings.Split(out, "\n") {
		parts := strings.Split(line, ":")
		if len(parts) != 2 {
			continue
		}
		if parts[0] == netType && strings.TrimSpace(parts[1]) == "connected" {
			return ConnectedStatus, nil
		}

		if parts[0] == netType && strings.TrimSpace(parts[1]) == "connecting" {
			return ConnectingStatus, nil
		}
	}
	return DisconnectedStatus, nil
}

func (nm *Manager) getInternetStatus(ctx context.Context) (bool, error) {
	out, err := nm.Run(ctx, "networking", "connectivity", "check")
	if err != nil {
		return false, fmt.Errorf("failed to query internet connectivity: %w", err)
	}

	return strings.TrimSpace(out) == "full", nil
}

func GetInternetStatus(ctx context.Context, conn remote.RemoteConn) (bool, error) {
	if conn == nil {
		return false, fmt.Errorf("missing connection")
	}
	nm := &Manager{
		Timeout: 5 * time.Second,
		Conn:    conn,
	}
	return nm.getInternetStatus(ctx)
}
