import { uniqueId } from 'lodash';

import {
  ArduinoBuilderBoard_BuilderApi,
  ArduinoBuilderBoardscomputev3_BuilderApi,
  ArduinoBuilderBoardsv3_BuilderApi,
  ArduinoBuilderBoardv3Full_BuilderApi,
  Compute_Response,
  GetBoardByFqbn_Response,
  GetBoards_Response,
  GetBoardsByVidPid_Response,
} from './builderApi.boards.type';
import {
  ArduinoBuilderCompilationResult_BuilderApi,
  ArduinoBuilderV2CompilationOutputResponse_BuilderApi,
  ArduinoBuilderV2CompilationsResponse_BuilderApi,
  CompileSketch_Response,
} from './builderApi.compilations.type';
import {
  ArduinoBuilderExample_BuilderApi,
  ArduinoBuilderExampleFile_BuilderApi,
  GetExampleFileContents_Response,
  GetExamples_Response,
} from './builderApi.examples.type';
import {
  GetFavoriteLibraries_Response,
  GetLibraries_Response,
  IsFavoriteLibrary,
  Libraries,
  LibrariesItem_Response,
  Library,
  LibraryDetails,
  LibraryDetails_Response,
} from './builderApi.libraries.type';
import {
  ArduinoBuilderExtrafile,
  ArduinoBuilderExtrafileV2,
} from './builderApi.type';

export function mapCompileSketchResponse(
  data: ArduinoBuilderCompilationResult_BuilderApi,
): CompileSketch_Response {
  return {
    bin: data.bin,
    binUrl: data.bin_url,
    elf: data.elf,
    elfUrl: data.elf_url,
    files: data.files,
    hex: data.hex,
    hexUrl: data.hex_url,
    otaKey: data.ota_key,
    stderr: data.stderr,
    stdout: data.stdout,
  };
}

export function normalizeV2toV1Extrafiles(
  data: ArduinoBuilderExtrafileV2[],
): ArduinoBuilderExtrafile[] {
  return data.map((item) => ({
    filename: item.filename,
    hex: item.data,
  }));
}

export function mapBuilderCompilationOutputNormalizeV2toV1Response(
  data: ArduinoBuilderV2CompilationOutputResponse_BuilderApi,
): CompileSketch_Response & { name: string } {
  return {
    name: data.name,
    bin: data?.bin,
    hex: data?.hex,
    elf: data?.elf,
    otaKey: data?.ota_key,
    files: data?.extra_files
      ? normalizeV2toV1Extrafiles(data?.extra_files)
      : undefined,
  };
}

export function mapSketchCompilationResponse(
  data: ArduinoBuilderV2CompilationsResponse_BuilderApi,
): ArduinoBuilderV2CompilationsResponse_BuilderApi {
  return data;
}

export function mapGetBoards(
  data: ArduinoBuilderBoardsv3_BuilderApi,
): GetBoards_Response {
  const boards = data.items.map((item) => ({
    id: uniqueId(),
    ...item,
  }));

  return {
    boards,
  };
}

export function mapGetBoardsByVidPid(
  data: ArduinoBuilderBoard_BuilderApi,
): GetBoardsByVidPid_Response {
  const firstItem = data && data.items && data.items[0];
  return {
    id: firstItem.board_id,
    name: firstItem.name,
    fqbn: firstItem.fqbn,
    architecture: firstItem.architecture,
  };
}

export function mapGetBoardByFqbn(
  data: ArduinoBuilderBoardv3Full_BuilderApi,
): GetBoardByFqbn_Response {
  return {
    id: data.board_id,
    architecture: data.architecture,
    fqbn: data.fqbn,
    name: data.name,
    menus: data.board_options,
    vendor: data.vendor,
  };
}

export function mapComputeUploadToBoardByFqbn(
  data: ArduinoBuilderBoardscomputev3_BuilderApi,
): Compute_Response {
  return {
    commandline: data.commandline,
    options: data.options,
    signature: data.signature,
    tools: data.tools,
    files: data.files,
  };
}

function mapLibrary(data: Library): LibrariesItem_Response {
  return {
    architectures: data.architectures,
    category: data.category,
    code: data.code,
    href: data.href,
    id: data.id,
    license: data.license,
    maintainer: data.maintainer,
    name: data.name,
    sentence: data.sentence,
    types: data.types,
    url: data.url,
    version: data.version,
    examplesNumber: data.examples_number,
    downloadUrl: data.download_url,
    isFavorite: IsFavoriteLibrary.Unknown,
  };
}

export function mapGetLibrariesResponse(
  data: Libraries,
): GetLibraries_Response {
  return {
    libraries: data.libraries.map(mapLibrary),
    totalItems: data.total_items,
    page: data.page,
    pages: data.pages,
  };
}

export function mapGetFavoriteLibrariesResponse(
  data: Library[],
): GetFavoriteLibraries_Response {
  return data.map((d) => ({
    ...mapLibrary(d),
    isFavorite: IsFavoriteLibrary.Yes,
  }));
}

export function mapGetLibraryResponse(
  data: LibraryDetails,
): LibraryDetails_Response {
  return {
    ...mapLibrary(data),
    files: data.files,
    examples: data.examples,
    otherVersions: data.other_versions,
  };
}

export function mapGetExamplesDataResponse(
  data: ArduinoBuilderExample_BuilderApi[],
): GetExamples_Response {
  return {
    examples: data,
  };
}

export function mapGetExampleFileContentsResponse(
  data: ArduinoBuilderExampleFile_BuilderApi,
): GetExampleFileContents_Response {
  return {
    name: data.name,
    data: data.data,
    mimetype: data.mimetype,
    href: data.href,
    path: data.path,
  };
}
