import { Config } from '@cloud-editor-mono/common';

import {
  httpDelete,
  httpGet,
  httpGetRaw,
  httpPost,
  httpPut,
} from '../fetch/fetch';
import { EventSourceHandlers, getEventSource } from '../fetch-event-source';
import {
  ArduinoBuilderBoard_BuilderApi,
  ArduinoBuilderBoardscomputev3_BuilderApi,
  ArduinoBuilderBoardsv3_BuilderApi,
  ArduinoBuilderBoardv3Full_BuilderApi,
  BoardsCompute_Body,
  BoardsCompute_Params,
  Compute_Response,
  GetBoardByFqbn_Params,
  GetBoardByFqbn_Response,
  GetBoards_Response,
  GetBoardsByVidPid_Params,
  GetBoardsByVidPid_Response,
} from './builderApi.boards.type';
import {
  ArduinoBuilderV2CancelCompilation_Params,
  ArduinoBuilderV2CompilationOutput_Params,
  ArduinoBuilderV2CompilationOutputResponse_BuilderApi,
  ArduinoBuilderV2CompilationsResponse_BuilderApi,
  ArduinoBuilderV2CompilationStream_Params,
  CompileSketch_Body,
  CompileSketch_Response,
} from './builderApi.compilations.type';
import {
  ArduinoBuilderExample_BuilderApi,
  ArduinoBuilderExampleFile_BuilderApi,
  GetExampleFileContents_Params,
  GetExampleFileContents_Response,
  GetExamples_Params,
  GetExamples_Response,
} from './builderApi.examples.type';
import {
  FavoriteLibrary_Params,
  GetFavoriteLibraries_Response,
  GetLibraries_Params,
  GetLibraries_Response,
  GetLibrary_Params,
  Libraries,
  Library,
  LibraryDetails,
  LibraryDetails_Response,
} from './builderApi.libraries.type';
import {
  mapBuilderCompilationOutputNormalizeV2toV1Response,
  mapComputeUploadToBoardByFqbn,
  mapGetBoardByFqbn,
  mapGetBoards,
  mapGetBoardsByVidPid,
  mapGetExampleFileContentsResponse,
  mapGetExamplesDataResponse,
  mapGetFavoriteLibrariesResponse,
  mapGetLibrariesResponse,
  mapGetLibraryResponse,
  mapSketchCompilationResponse,
} from './mapper';

// const BOARDS_ENDPOINT = '/boards/v1/boards';

export function builderAliveRequest(token: string): Promise<Response | void> {
  const endpoint = '/alive';
  return httpGetRaw(Config.BUILDER_API_V2_URL, endpoint, token);
}

export async function getBoardsRequest(): Promise<GetBoards_Response> {
  const response = await httpGet<ArduinoBuilderBoardsv3_BuilderApi>(
    Config.BOARDS_API_URL,
    undefined,
    '',
  );

  if (!response) {
    throw new Error(
      `Call to "${Config.BUILDER_API_URL}" did not respond with the expected result`,
    );
  }

  return mapGetBoards(response);
}

export async function getBoardsByVidPidRequest(
  params: GetBoardsByVidPid_Params,
): Promise<GetBoardsByVidPid_Response> {
  const endpoint = '';
  const vidPid = `${params.vid}-${params.pid}`;
  const requestParams = { 'vid-pid': vidPid };
  const response = await httpGet<ArduinoBuilderBoard_BuilderApi>(
    Config.BOARDS_API_URL,
    undefined,
    endpoint,
    undefined,
    requestParams,
  );

  if (!response) {
    throw new Error(
      `Call to "${Config.API_URL}" with params "${vidPid}" did not respond with the expected result`,
    );
  }

  return mapGetBoardsByVidPid(response);
}

export async function getBoardByFqbnRequest(
  params: GetBoardByFqbn_Params,
): Promise<GetBoardByFqbn_Response> {
  const endpoint = `/${params.fqbn}`;
  const response = await httpGet<ArduinoBuilderBoardv3Full_BuilderApi>(
    Config.BOARDS_API_URL,
    undefined,
    endpoint,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetBoardByFqbn(response);
}

export async function computeActionByFqbnRequest(
  params: BoardsCompute_Params,
  payload: BoardsCompute_Body,
): Promise<Compute_Response> {
  const endpoint = `/${params.fqbn}/upload-command`;
  const response = await httpGet<ArduinoBuilderBoardscomputev3_BuilderApi>(
    Config.BOARDS_API_URL,
    undefined,
    endpoint,
    undefined,
    payload,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapComputeUploadToBoardByFqbn(response);
}

export async function getLibrariesRequest(
  params: GetLibraries_Params,
): Promise<GetLibraries_Response> {
  const endpoint = '/v1/libraries';

  const response = await httpGet<Libraries>(
    Config.BUILDER_API_URL,
    undefined,
    endpoint,
    undefined,
    params,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetLibrariesResponse(response);
}

export async function getLibraryRequest(
  params: GetLibrary_Params,
): Promise<LibraryDetails_Response> {
  const endpoint = `/v1/libraries/${params.id}`;

  const response = await httpGet<LibraryDetails>(
    Config.BUILDER_API_URL,
    undefined,
    endpoint,
    undefined,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetLibraryResponse(response);
}

export async function getExamplesRequest(
  params: GetExamples_Params,
): Promise<GetExamples_Response> {
  const endpoint = '/v1/examples';

  const response = await httpGet<ArduinoBuilderExample_BuilderApi[]>(
    Config.BUILDER_API_URL,
    undefined,
    endpoint,
    undefined,
    params,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetExamplesDataResponse(response);
}

export async function getFavoriteLibrariesRequest(
  token: string,
): Promise<GetFavoriteLibraries_Response> {
  const endpoint = '/pinned';

  const response = await httpGet<Library[]>(
    Config.BUILDER_API_URL,
    undefined,
    endpoint,
    token,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetFavoriteLibrariesResponse(response);
}

export function addFavoriteLibraryRequest(
  params: FavoriteLibrary_Params,
  token: string,
): Promise<void> {
  const endpoint = `/pinned/${params.id}`;

  return httpPut(Config.BUILDER_API_URL, undefined, endpoint, undefined, token);
}

export function removeFavoriteLibraryRequest(
  params: FavoriteLibrary_Params,
  token: string,
): Promise<void> {
  const endpoint = `/pinned/${params.id}`;

  return httpDelete(Config.BUILDER_API_URL, undefined, endpoint, token);
}

export async function getBuilderFileContentsRequest(
  params: GetExampleFileContents_Params,
): Promise<GetExampleFileContents_Response> {
  const endpoint = `/v1/files/${params.path}`;
  const response = await httpGet<ArduinoBuilderExampleFile_BuilderApi>(
    Config.BUILDER_API_URL,
    undefined,
    endpoint,
  );

  if (!response) {
    throw new Error(
      `Call to "${endpoint}" did not respond with the expected result`,
    );
  }

  return mapGetExampleFileContentsResponse(response);
}



const BASE_COMPILATION_URL = '/v1/compilations';
export async function postCreateSketchCompilationRequest(
  body: CompileSketch_Body,
  token: string,
  abortController?: AbortController,
): Promise<ArduinoBuilderV2CompilationsResponse_BuilderApi> {
  const endpoint = BASE_COMPILATION_URL;
  const response =
    await httpPost<ArduinoBuilderV2CompilationsResponse_BuilderApi>(
      Config.BUILDER_API_V2_URL,
      undefined,
      endpoint,
      body,
      token,
      abortController,
      
    );

  if (response) {
    return mapSketchCompilationResponse(response);
  } else {
    if (abortController?.signal.aborted) {
      throw new Error('Create Sketch compilation Aborted');
    }
    throw new Error('Create Sketch compilation Error');
  }
}

export async function getCreatedSketchCompilationRequest(
  params: ArduinoBuilderV2CompilationStream_Params,
  token: string,
): Promise<ArduinoBuilderV2CompilationsResponse_BuilderApi> {
  const endpoint = `${BASE_COMPILATION_URL}/${params.id}`;
  const response =
    await httpGet<ArduinoBuilderV2CompilationsResponse_BuilderApi>(
      Config.BUILDER_API_V2_URL,
      undefined,
      endpoint,
      token,
      undefined,
      
    );

  if (response) {
    return mapSketchCompilationResponse(response);
  } else {
    throw new Error('Get Create Sketch compilation Error');
  }
}

export async function postCancelSketchCompilationRequest(
  params: ArduinoBuilderV2CancelCompilation_Params,
  token: string,
): Promise<ArduinoBuilderV2CompilationsResponse_BuilderApi> {
  const endpoint = `${BASE_COMPILATION_URL}/${params.id}/cancel`;
  const response =
    await httpPost<ArduinoBuilderV2CompilationsResponse_BuilderApi>(
      Config.BUILDER_API_V2_URL,
      undefined,
      endpoint,
      undefined,
      token,
      undefined,
      
    );

  if (response && response.status === 'cancelled') {
    return mapSketchCompilationResponse(response);
  } else {
    throw new Error('Cancel Sketch compilation Error');
  }
}

export async function getBuilderCompilationOutputRequest(
  params: ArduinoBuilderV2CompilationOutput_Params,
  token: string,
): Promise<CompileSketch_Response & { name: string }> {
  const endpoint = `${BASE_COMPILATION_URL}/${params.id}/artifacts`;
  const response =
    await httpGet<ArduinoBuilderV2CompilationOutputResponse_BuilderApi>(
      Config.BUILDER_API_V2_URL,
      undefined,
      endpoint,
      token,
      params.type
        ? {
            type: params.type,
          }
        : undefined,
      
    );

  if (response) {
    return mapBuilderCompilationOutputNormalizeV2toV1Response(response);
  } else {
    throw new Error('Compilation output Error');
  }
}

export async function getCompilationStreamRequest(
  params: ArduinoBuilderV2CompilationStream_Params,
  handlers: EventSourceHandlers,
  token: string,
  abortController?: AbortController,
): Promise<void> {
  const endpoint = `${BASE_COMPILATION_URL}/${params.id}/events`;
  const sseUrl = `${Config.BUILDER_API_V2_URL}${endpoint}`;

  return getEventSource(
    sseUrl,
    handlers,
    token,
    abortController,
    
  );
}
