import {
  findUIPort,
  getApps,
  openUIWhenReady,
  startApp,
  stopApp,
} from '@cloud-editor-mono/domain/src/services/services-by-app/app-lab';
import { AppDetailedInfo, ErrorData } from '@cloud-editor-mono/infrastructure';
import {
  AL_STARTUP_KEY,
  AppLabAction,
  AppLabActionStatus,
  UseRuntimeLogic,
} from '@cloud-editor-mono/ui-components/lib/components-by-app/app-lab';
import { useQueryClient } from '@tanstack/react-query';
import { useCallback, useState } from 'react';

import { useAppSSE } from '../../features/app-detail/hooks/useAppSSE';
import { useConsoleSources } from '../../features/app-detail/hooks/useConsoleSources';
import { useCurrentAction } from '../../features/app-detail/hooks/useCurrentAction';
import { RuntimeContextValue } from './runtimeContext';

export const useRuntimeLogic: UseRuntimeLogic =
  function (): RuntimeContextValue {
    const queryClient = useQueryClient();
    const {
      currentAction,
      currentActionStatus,
      send: sendCurrentAction,
    } = useCurrentAction();

    const [appIdToStop, setAppIdToStop] = useState<string | null>(null);
    const [activeApp, setActiveApp] = useState<AppDetailedInfo | undefined>();

    const {
      consoleTabs,
      consoleSources,
      activeConsoleTab,
      addConsoleSource,
      setActiveConsoleTab,
      consoleSourcesOwner,
      consoleSourcesResetSubject,
      appendDataToSource: appendData,
      reset: resetConsoleSources,
    } = useConsoleSources();

    //On startup success if called after start/stop actions succeeded.
    const onStartupSuccess = async (): Promise<void> => {
      appendData(AL_STARTUP_KEY, undefined, undefined, {
        className: 'success',
        isGlobalStyle: true,
      });
      sendCurrentAction({
        type: 'ACTION_SUCCEEDED',
      });

      //Invalidates two queries. ['sync-app'] and ['sync-app', appId]
      queryClient.invalidateQueries({
        queryKey: ['sync-app'],
      });
    };

    //On startup success if called after start/stop actions fail.
    const onStartupError = (data?: ErrorData): void => {
      appendData(AL_STARTUP_KEY, data, undefined, {
        className: 'error',
      });
      sendCurrentAction({
        type: 'ACTION_FAILED',
      });
    };

    const {
      connect: startAppStream,
      abort: startAppAbort,
      progress: startAppProgress,
    } = useAppSSE({
      appSSE: startApp,
      onMessage: (message): void => {
        appendData(AL_STARTUP_KEY, message, true);
      },
      onSuccess: onStartupSuccess,
      onError: onStartupError,
    });

    //On stop app success we need to check if there was another app stopping, if that's the case we swap apps.
    const onStopAppSuccess = (): void => {
      if (appIdToStop && activeApp?.id) {
        //Swap application flow
        sendCurrentAction({
          type: 'ACTION_REQUESTED',
          payload: {
            currentAction: AppLabAction.Run,
          },
        });
        startAppStream(activeApp.id);
        setAppIdToStop(null);
      } else {
        onStartupSuccess();
      }

      setActiveApp(undefined);
    };

    const {
      connect: stopAppStream,
      abort: stopAppAbort,
      progress: stopAppProgress,
    } = useAppSSE({
      appSSE: stopApp,
      onMessage: (message): void => {
        appendData(AL_STARTUP_KEY, message, true);
      },
      onSuccess: onStopAppSuccess,
      onError: onStartupError,
    });

    const resetStreams = useCallback((): void => {
      startAppAbort();
      stopAppAbort();
    }, [startAppAbort, stopAppAbort]);

    const cleanUp = useCallback((): void => {
      resetStreams();
      sendCurrentAction({ type: 'RESET' });

      consoleSourcesResetSubject.next();
      resetConsoleSources([]);

      setActiveApp(undefined);
      setAppIdToStop(null);
    }, [
      consoleSourcesResetSubject,
      resetConsoleSources,
      resetStreams,
      sendCurrentAction,
    ]);

    const runAction = useCallback(
      async (
        app: AppDetailedInfo,
        displaySwapDialog?: (e: boolean) => void,
      ): Promise<void> => {
        cleanUp();

        if (displaySwapDialog) {
          //Refetch and get the latest data
          const response = await getApps({ query: { status: 'running' } });
          const isAnotherAppRunning = response?.some(
            (runningApp) => runningApp.id !== app.id,
          );

          if (isAnotherAppRunning) {
            const idToStop = response.find(
              (runningApp) => runningApp.id !== app.id,
            )?.id;
            if (!idToStop) return;

            displaySwapDialog(true);
            setAppIdToStop(idToStop);
            return;
          }
        }

        //Run action start
        setActiveApp(app);
        addConsoleSource(AL_STARTUP_KEY, { sourcesOwnerAppId: app.id });
        setActiveConsoleTab(AL_STARTUP_KEY);
        startAppStream(app.id);

        sendCurrentAction({
          type: 'ACTION_REQUESTED',
          payload: {
            currentAction: AppLabAction.Run,
          },
        });

        try {
          const uiPort = await findUIPort(app.id);
          await openUIWhenReady(uiPort);
        } catch (error) {
          console.error(`Failed to open UI for app ${app.id}:`, error);
        }
      },
      [
        addConsoleSource,
        cleanUp,
        sendCurrentAction,
        setActiveConsoleTab,
        startAppStream,
      ],
    );

    const stopAction = useCallback(
      async (app: AppDetailedInfo) => {
        addConsoleSource(AL_STARTUP_KEY, { sourcesOwnerAppId: app.id });
        setActiveConsoleTab(AL_STARTUP_KEY);

        if (!activeApp?.id) {
          setActiveApp(app);
        }

        //Aborting the previous action
        if (
          currentActionStatus === AppLabActionStatus.Pending &&
          app.status !== 'running'
        ) {
          sendCurrentAction({
            type: 'ACTION_SUCCEEDED',
            payload: {
              currentAction: AppLabAction.Stop,
            },
          });
          return;
        }

        sendCurrentAction({
          type: 'ACTION_REQUESTED',
          payload: {
            currentAction: AppLabAction.Stop,
          },
        });

        stopAppStream(app.id);
      },
      [
        addConsoleSource,
        setActiveConsoleTab,
        activeApp?.id,
        currentActionStatus,
        sendCurrentAction,
        stopAppStream,
      ],
    );

    const swapAction = useCallback(
      async (app: AppDetailedInfo): Promise<void> => {
        if (!appIdToStop || !app.id) return;
        resetConsoleSources([]);

        addConsoleSource(AL_STARTUP_KEY, { sourcesOwnerAppId: app.id });
        setActiveConsoleTab(AL_STARTUP_KEY);

        setActiveApp(app);
        sendCurrentAction({
          type: 'ACTION_REQUESTED',
          payload: {
            currentAction: AppLabAction.Stop,
          },
        });
        stopAppStream(appIdToStop);
      },
      [
        addConsoleSource,
        appIdToStop,
        resetConsoleSources,
        sendCurrentAction,
        setActiveConsoleTab,
        stopAppStream,
      ],
    );

    const resetCurrentAction = useCallback((): void => {
      sendCurrentAction({ type: 'RESET' });
    }, [sendCurrentAction]);

    return {
      activeApp,
      runAction,
      stopAction,
      swapAction,
      consoleSourcesResetSubject,
      resetCurrentAction: resetCurrentAction,
      resetConsoleSources,
      currentAction,
      currentActionStatus,
      consoleSources,
      progress: stopAppProgress || startAppProgress,
      consoleTabs,
      activeConsoleTab,
      setActiveConsoleTab,
      appendData,
      addConsoleSource,
      consoleSourcesOwner,
    };
  };
