import {
  checkAndApplyUpdate,
  getApps,
  getCurrentVersion,
  getSystemResources,
  newVersion,
} from '@cloud-editor-mono/domain/src/services/services-by-app/app-lab';
import {
  ArduinoLoop,
  // WiFiDisconnected,
} from '@cloud-editor-mono/images/assets/icons';
import {
  AppDetailedInfo,
  SystemResourcesStreamMessageType,
} from '@cloud-editor-mono/infrastructure';
import {
  FooterItem,
  Notification,
} from '@cloud-editor-mono/ui-components/lib/app-lab-footer-bar/AppLabFooterBar.type';
import {
  FooterBarLogic,
  useI18n,
} from '@cloud-editor-mono/ui-components/lib/components-by-app/app-lab';
import { useQuery } from '@tanstack/react-query';
import { useContext, useEffect, useRef, useState } from 'react';

import { BoardConfigurationContext } from '../../providers/board-configuration/boardConfigurationContext';
import { NetworkContext } from '../../providers/network/networkContext';
import { RuntimeContext } from '../../providers/runtime/runtimeContext';
import { SetupContext } from '../../providers/setup/setupContext';
import { useBoardLifecycleStore } from '../../store/boards/boards';
import { useCreateAppTitleLogic } from '../app-detail/appDetailTitle.logic';
import { messages } from './messages';

const placeholderFooterItems: FooterItem[] = [
  {
    id: 'board',
  },
  {
    id: 'root',
  },
  {
    id: 'user',
  },
  {
    id: 'ram',
  },
  {
    id: 'cpu',
  },
  {
    id: 'network',
  },
  {
    id: 'ip',
  },
];

const bytesToGiB = (bytes: unknown): string =>
  ((bytes as number) / 1024 / 1024 / 1024).toFixed(2);

const getUsedPercent = (used: unknown, total: unknown): number =>
  ((used as number) / (total as number)) * 100;

// Temporarily disable footer update notifications in favour of BoardUpdateDialog
const enableFooterUpdate = false;

export const useFooterBarLogic: FooterBarLogic =
  function (): ReturnType<FooterBarLogic> {
    const { formatMessage } = useI18n();
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const [items, setItems] = useState<FooterItem[]>([
      ...placeholderFooterItems,
    ]);
    const [notifications, setNotifications] = useState<Notification[]>([]);
    const [newNotifications, setNewNotifications] = useState<number>(0);
    const abortControllerStream = useRef<AbortController>();

    const runtimeContext = useContext(RuntimeContext);

    const { boardIsReachable, selectedConnectedBoard: selectedBoard } =
      useBoardLifecycleStore();

    const { data: apps } = useQuery(
      ['sync-app'],
      () => {
        return getApps({});
        //return getApps({ query: { filter: 'default' } });
      },
      {
        enabled: boardIsReachable,
        refetchOnWindowFocus: false,
      },
    );

    const { data: currentVersion } = useQuery(
      ['current-version'],
      () => getCurrentVersion(),
      {
        refetchOnWindowFocus: false,
      },
    );

    useEffect(() => {
      enableFooterUpdate &&
        newVersion()
          .then((v: string) => {
            if (v !== '') {
              console.log('New version available:', v);
              setNewNotifications((prev) => prev + 1);
              setNotifications((prev) => [
                ...prev,
                {
                  icon: <ArduinoLoop />,
                  label: formatMessage(messages.updateAvailable),
                  tooltip: formatMessage(messages.updateAvailableTooltip, {
                    v,
                  }),
                  onClick: checkAndApplyUpdate,
                },
              ]);
            }
          })
          .catch((error: Error) => {
            console.error('Error calling NewVersion:', error);
          });

      return () => {
        abortControllerStream.current?.abort();
      };

      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);

    useEffect(() => {
      const openStream = (): Promise<void> =>
        getSystemResources(
          {
            onopen: undefined,
            onclose: () => {
              abortControllerStream.current?.abort();
              abortControllerStream.current = new AbortController();

              setTimeout(() => {
                openStream();
              }, 3000);
            },
            onerror: undefined,
            onmessage: (event) => {
              const messageType = event.event;
              const data = JSON.parse(event.data);

              setItems((prev) =>
                prev.map((item) => {
                  if (
                    messageType === SystemResourcesStreamMessageType.Cpu &&
                    item.id === 'cpu'
                  ) {
                    return {
                      ...item,
                      label: formatMessage(messages.cpu, {
                        used: (data.used_percent as number).toFixed(0),
                      }),
                      state: data.used_percent > 80 ? 'warning' : undefined,
                    };
                  }
                  if (
                    messageType === SystemResourcesStreamMessageType.Memory &&
                    item.id === 'ram'
                  ) {
                    return {
                      ...item,
                      label: formatMessage(messages.memory, {
                        used: bytesToGiB(data.used),
                        total: bytesToGiB(data.total),
                      }),
                      state:
                        getUsedPercent(data.used, data.total) > 80
                          ? 'warning'
                          : undefined,
                    };
                  }
                  if (
                    messageType === SystemResourcesStreamMessageType.Disk &&
                    item.id === 'user' &&
                    data.path === '/home/arduino' // return only home directory disk usage
                  ) {
                    return {
                      ...item,
                      label: formatMessage(messages.disk, {
                        path: 'USER',
                        used: bytesToGiB(data.used),
                        total: bytesToGiB(data.total),
                      }),
                      state:
                        getUsedPercent(data.used, data.total) > 90
                          ? 'warning'
                          : undefined,
                    };
                  }
                  if (
                    messageType === SystemResourcesStreamMessageType.Disk &&
                    item.id === 'root' &&
                    data.path === '/'
                  ) {
                    return {
                      ...item,
                      label: formatMessage(messages.disk, {
                        path: 'ROOT',
                        used: bytesToGiB(data.used),
                        total: bytesToGiB(data.total),
                      }),
                      state:
                        getUsedPercent(data.used, data.total) > 90
                          ? 'warning'
                          : undefined,
                    };
                  }
                  return item;
                }),
              );
            },
          },
          abortControllerStream.current,
        );

      if (boardIsReachable) {
        openStream();
      }
    }, [formatMessage, boardIsReachable]);

    const resetNewNotifications = (): void => {
      setNewNotifications(0);
    };

    const { boardConfigurationIsSet, boardName } = useContext(
      BoardConfigurationContext,
    );

    const { isConnected } = useContext(NetworkContext);

    useEffect(() => {
      setItems((prev) =>
        prev.map((item) => {
          if (item.id === 'board') {
            if (boardIsReachable) {
              return {
                ...item,
                label: boardConfigurationIsSet
                  ? `${boardName} (Arduino UNO Q)`
                  : 'Arduino UNO Q',
                state: 'default',
              };
            }
            return {
              id: 'board',
              state: 'inactive',
            };
          }
          return item;
        }),
      );
    }, [boardName, boardIsReachable, boardConfigurationIsSet]);

    useEffect(() => {
      setItems((prev) =>
        prev.map((item) => {
          if (item.id === 'ip') {
            if (
              boardIsReachable &&
              selectedBoard &&
              (selectedBoard.serial || selectedBoard.address)
            ) {
              return {
                ...item,
                label: formatMessage(messages.ip, {
                  type: selectedBoard.serial ? 'SERIAL' : 'IP',
                  ip: selectedBoard.serial || selectedBoard.address,
                }),
              };
            }
          }
          return item;
        }),
      );
    }, [boardIsReachable, formatMessage, selectedBoard]);

    const { setSetupCompleted } = useContext(SetupContext);
    useEffect(() => {
      if (boardIsReachable) {
        setItems((prev) =>
          prev.map((item) => {
            if (item.id === 'network') {
              return {
                ...item,
                state: isConnected ? 'default' : 'inactive',
                onClick: !isConnected
                  ? (): void => {
                      setSetupCompleted(false);
                    }
                  : undefined,
              };
            }
            return item;
          }),
        );
      }
    }, [boardIsReachable, isConnected, setSetupCompleted]);

    return {
      defaultApp: apps?.find((app) => app.default) as AppDetailedInfo,
      runningApp: apps?.find(
        (app) => app.status === 'running',
      ) as AppDetailedInfo,
      runtimeContext,
      currentVersion: currentVersion || '',
      notifications,
      newNotifications,
      resetNewNotifications,
      items,
      useCreateAppTitleLogic,
    };
  };
