import {
  AddBrick as AddBrickIcon,
  AddLibrary as AddLibraryIcon,
  CaretDown as CaretDownIcon,
  FileAdd as FileAddIcon,
} from '@cloud-editor-mono/images/assets/icons';
import {
  AddAppBrickDialog,
  AddSketchLibraryDialog,
  AddSketchLibraryDialogLogic,
  AppLabBrickItem,
  AppLabLibraryItem,
  Button,
  ButtonSize,
  ButtonType,
  ButtonVariant,
  DeleteAppBrickDialog,
  FileNode,
  FileTree,
  FileTreeApi,
  FolderNode,
  isFileNode,
  SelectableFileData,
  TreeNode,
} from '@cloud-editor-mono/ui-components/lib/components-by-app/app-lab';
import clsx from 'clsx';
import Split from 'react-split';
7;
import {
  AppDetailedInfo,
  BrickInstance,
  BrickListItem,
} from '@cloud-editor-mono/infrastructure';
import { useRef, useState } from 'react';
import { useMeasure } from 'react-use';

import { AppsSection } from '../../../routes/__root';
import EditorFeat from '../../editor/Editor.feat';
import { EditorLogicParams } from '../../editor/editor.type';
import styles from '../app-detail.module.scss';
import { useAppFilesSectionLogic } from './AppFilesSection.logic';

interface AppFilesSection {
  app: AppDetailedInfo | undefined;
  appBricks: BrickInstance[] | undefined;
  bricks: BrickListItem[] | undefined;
  appLibraries?: Array<{ id: string; version: string }>;
  section: AppsSection;
  fileTree?: TreeNode[];
  selectedFile?: SelectableFileData;
  selectedNode?: FileNode;
  defaultOpenFoldersState: { [key: string]: boolean } | undefined;
  setSelectedFile: (id: string | TreeNode | undefined) => void;
  openFilesFolder: () => void;
  openExternal: () => void;
  openExternalLink: (url: string) => void;
  addAppBrick(brickId: string): Promise<boolean>;
  deleteAppBrick(brickId: string): Promise<boolean>;
  editorLogicParams: EditorLogicParams;
  addFileHandler: (path: string) => Promise<void>;
  renameFileHandler: (path: string, newName: string) => Promise<void>;
  deleteFileHandler: (path: string) => Promise<void>;
  addSketchLibraryDialogLogic: AddSketchLibraryDialogLogic;
  openAddSketchLibraryDialog: () => void;
  deleteSketchLibrary: (libRef: string) => Promise<void>;
}

const AppFilesSection: React.FC<AppFilesSection> = (props: AppFilesSection) => {
  const {
    appBricks,
    bricks,
    appLibraries,
    section,
    fileTree,
    selectedFile,
    selectedNode,
    defaultOpenFoldersState,
    setSelectedFile,
    openExternalLink,
    addAppBrick,
    deleteAppBrick,
    editorLogicParams,
    addFileHandler,
    renameFileHandler,
    deleteFileHandler,
    addSketchLibraryDialogLogic,
    openAddSketchLibraryDialog,
    deleteSketchLibrary,
  } = props;

  const [ref, { height }] = useMeasure<HTMLDivElement>();
  const {
    addAppBrickDialogLogic,
    openAddAppBrickDialog,
    deleteAppBrickDialogLogic,
    openDeleteAppBrickDialog,
  } = useAppFilesSectionLogic({
    appBricks,
    bricks,
    addAppBrick,
    deleteAppBrick,
    openExternalLink,
  });

  const [collapseBricks, setCollapseBricks] = useState(false);
  const [collapseLibraries, setCollapseLibraries] = useState(false);
  const [collapseFiles, setCollapseFiles] = useState(false);

  // Omit the root folder
  const files =
    fileTree && fileTree[0] ? (fileTree[0] as FolderNode).children : undefined;

  const fileTreeRef = useRef<FileTreeApi>(null);

  return (
    <Split
      className={styles['split']}
      sizes={[20, 80]}
      minSize={200}
      expandToMin={false}
      gutterSize={20}
      gutterAlign="center"
      snapOffset={30}
      direction="horizontal"
      cursor="col-resize"
      gutter={(): HTMLElement => {
        const element = document.createElement('div');
        element.className = styles['gutter'];
        return element;
      }}
    >
      <AddAppBrickDialog logic={addAppBrickDialogLogic} />
      <DeleteAppBrickDialog logic={deleteAppBrickDialogLogic} />
      <div className={(styles['split-item'], styles['split-item-left'])}>
        <div className={styles['app-bricks']}>
          <div className={styles['app-header']}>
            <Button
              onClick={(): void => {
                setCollapseBricks((prev) => !prev);
              }}
              size={ButtonSize.XXSmall}
              classes={{
                button: clsx(
                  styles['app-header-title'],
                  collapseBricks && styles['collapsed'],
                ),
              }}
            >
              <CaretDownIcon />
              Bricks
            </Button>
            {section === 'my-apps' && (
              <Button
                bold
                onClick={openAddAppBrickDialog}
                type={ButtonType.Secondary}
                size={ButtonSize.XSmall}
                Icon={AddBrickIcon}
                variant={ButtonVariant.LowContrast}
                classes={{
                  button: styles['app-header-button'],
                }}
              />
            )}
          </div>
          {!collapseBricks && (
            <>
              {appBricks && appBricks.length > 0 && (
                <div className={styles['app-list']}>
                  {appBricks.map((brick) => (
                    <AppLabBrickItem
                      key={brick.id}
                      brick={brick}
                      selected={brick.id === selectedFile?.fileId}
                      onClick={(): void => setSelectedFile(brick.id ?? '')}
                      onDelete={
                        section === 'my-apps'
                          ? (): void => openDeleteAppBrickDialog(brick)
                          : undefined
                      }
                    />
                  ))}
                </div>
              )}
              {appBricks && !appBricks.length && (
                <p className={styles['app-empty']}>No bricks added yet</p>
              )}
            </>
          )}
        </div>

        <AddSketchLibraryDialog logic={addSketchLibraryDialogLogic} />
        <div className={styles['app-libraries']}>
          <div className={styles['app-header']}>
            <Button
              onClick={(): void => {
                setCollapseLibraries((prev) => !prev);
              }}
              size={ButtonSize.XXSmall}
              classes={{
                button: clsx(
                  styles['app-header-title'],
                  collapseBricks && styles['collapsed'],
                ),
              }}
            >
              <CaretDownIcon />
              Libraries
            </Button>
            {section === 'my-apps' && (
              <Button
                bold
                onClick={openAddSketchLibraryDialog}
                type={ButtonType.Secondary}
                size={ButtonSize.XSmall}
                Icon={AddLibraryIcon}
                variant={ButtonVariant.LowContrast}
                classes={{
                  button: styles['app-header-button'],
                }}
              />
            )}
          </div>
          {!collapseLibraries && (
            <>
              {appLibraries && appLibraries.length > 0 && (
                <div className={styles['app-list']}>
                  {appLibraries.map((lib) => (
                    <AppLabLibraryItem
                      key={lib.id}
                      name={lib.id}
                      version={lib.version}
                      onDelete={
                        section === 'my-apps'
                          ? (): Promise<void> =>
                              deleteSketchLibrary(`${lib.id}@${lib.version}`)
                          : undefined
                      }
                    />
                  ))}
                </div>
              )}
              {appLibraries && !appLibraries.length && (
                <p className={styles['app-empty']}>
                  No sketch libraries added yet
                </p>
              )}
            </>
          )}
        </div>

        <div className={styles['app-files']}>
          <div className={styles['app-header']}>
            <Button
              onClick={(): void => {
                setCollapseFiles((prev) => !prev);
              }}
              size={ButtonSize.XXSmall}
              classes={{
                button: clsx(
                  styles['app-header-title'],
                  collapseFiles && styles['collapsed'],
                ),
              }}
            >
              <CaretDownIcon />
              Files
            </Button>
            {section === 'my-apps' && (
              <Button
                onClick={(): void => {
                  fileTreeRef.current?.handleFileCreate();
                }}
                type={ButtonType.Secondary}
                size={ButtonSize.XSmall}
                Icon={FileAddIcon}
                variant={ButtonVariant.LowContrast}
                classes={{
                  button: clsx(styles['app-header-button']),
                }}
              />
            )}
          </div>
          {!collapseFiles && (
            <div ref={ref} className={styles['app-list']}>
              <FileTree
                ref={fileTreeRef}
                height={height}
                nodes={files}
                selectedNode={selectedNode}
                selectedFileChange={setSelectedFile}
                defaultOpenFoldersState={defaultOpenFoldersState}
                onFileCreate={addFileHandler}
                onFileRename={renameFileHandler}
                onFileDelete={deleteFileHandler}
                isReadOnly={section !== 'my-apps'}
                isBricksSelected={selectedFile?.fileExtension === 'brick'}
              />
            </div>
          )}
        </div>
      </div>
      <div className={clsx(styles['split-item'], styles['split-item-right'])}>
        {((selectedNode &&
          isFileNode(selectedNode) &&
          !!editorLogicParams.openFiles.length) ||
          selectedFile?.fileExtension === 'brick') && (
          <EditorFeat editorLogicParams={editorLogicParams} />
        )}
      </div>
    </Split>
  );
};

export default AppFilesSection;
