import { WebSocketHandlers } from '@cloud-editor-mono/infrastructure';
import { useEffect, useRef, useState } from 'react';

interface UseWebSocket {
  connect: () => Promise<void>;
  abort: () => void;
  send: (message: string) => void;
  error: string | null;
  isConnected: boolean;
}

export type AppWebSocket = (handlers: WebSocketHandlers) => Promise<WebSocket>;

export const useAppWebSocket = ({
  appWebSocket,
  handlers,
  onMessage,
}: {
  appWebSocket: AppWebSocket;
  handlers?: WebSocketHandlers;
  onMessage: (data: string) => void;
}): UseWebSocket => {
  const [error, setError] = useState<string | null>(null);
  const abortController = useRef<AbortController | null>(null);
  const [isConnected, setIsConnected] = useState<boolean>(true);
  const socketRef = useRef<WebSocket | null>(null);

  const onmessage: WebSocketHandlers['onmessage'] = async (
    event: MessageEvent,
  ) => {
    const message = await (event.data as Blob).text();

    onMessage(message);
    handlers?.onmessage?.(event);
  };

  const onerror: WebSocketHandlers['onerror'] = (error) => {
    setError(`An error occurred: ${error}`);
    handlers?.onerror?.(error);
  };

  const onopen: WebSocketHandlers['onopen'] = (response) => {
    setIsConnected(true);
    handlers?.onopen?.(response);
  };

  const onclose: WebSocketHandlers['onclose'] = () => {
    setIsConnected(false);
    handlers?.onclose?.();
  };

  const abort = (): void => {
    if (socketRef.current) {
      socketRef.current.close();
      socketRef.current = null;
    }

    setIsConnected(false);
    setError(null);
  };

  useEffect(() => {
    return () => {
      abort();
    };
  }, []);

  const connect = async (): Promise<void> => {
    //Clean old states
    abort();

    abortController.current = new AbortController();
    try {
      socketRef.current = await appWebSocket({
        onopen,
        onmessage,
        onerror,
        onclose,
      });
    } catch (err) {
      console.error(err);
      setError(
        err instanceof Error
          ? err.message
          : 'Unknown WebSocket error when connecting',
      );
    }
  };

  const send = (message: string): void => {
    if (socketRef.current) {
      socketRef.current.send(message);
    }
  };

  return {
    connect,
    abort,
    send,
    error,
    isConnected,
  };
};
