import {
  ErrorData,
  EventSourceHandlers,
  MessageData,
  StreamEvent,
  StreamEventType,
} from '@cloud-editor-mono/infrastructure';
import { EventSourceMessage } from '@microsoft/fetch-event-source';
import { useCallback, useEffect, useRef, useState } from 'react';

interface UseSSE {
  connect: (id: string) => Promise<void>;
  abort: () => void;
  progress: number;
  error: string | null;
  isConnected: boolean;
}

export type AppSSE = (
  id: string,
  handlers: EventSourceHandlers,
  abortController?: AbortController,
) => Promise<void>;

export const useAppSSE = ({
  appSSE,
  handlers,
  onSuccess,
  onError,
  onMessage,
}: {
  appSSE: AppSSE;
  handlers?: EventSourceHandlers;
  onSuccess?: () => void;
  onError?: (data?: ErrorData) => void;
  onMessage: (data: MessageData) => void;
}): UseSSE => {
  const [progress, setProgress] = useState<number>(0);
  const [error, setError] = useState<string | null>(null);
  const abortController = useRef<AbortController | null>(null);
  const [isConnected, setIsConnected] = useState<boolean>(true);
  const [serverClosed, setServerClosed] = useState<boolean>(false);

  const abort = useCallback((): void => {
    if (abortController.current) {
      abortController.current.abort();
      abortController.current = null;
    }

    setIsConnected(false);
    setProgress(0);
    setError(null);
  }, []);

  useEffect(() => {
    if (
      progress === 100 &&
      !error &&
      serverClosed &&
      !isConnected &&
      onSuccess
    ) {
      onSuccess();
      abort();
    }
  }, [error, isConnected, onSuccess, progress, serverClosed, abort]);

  const onmessage: EventSourceHandlers['onmessage'] = useCallback(
    (event: EventSourceMessage) => {
      let normalizedEvent: StreamEvent;
      try {
        const parsedData = JSON.parse(event.data);
        normalizedEvent = {
          event: event.event as StreamEventType,
          data: parsedData,
        };
      } catch (parseError) {
        console.warn(parseError);
        return;
      }

      if (normalizedEvent.event === StreamEventType.Progress) {
        setProgress(normalizedEvent.data.progress);
      }

      if (normalizedEvent.event === StreamEventType.Error) {
        if (normalizedEvent.data.code === 'SERVER_CLOSED') {
          setServerClosed(true);
          return;
        }

        setError(normalizedEvent.data.code);
        onError?.(normalizedEvent.data);
      }

      if (normalizedEvent.event === StreamEventType.Message) {
        onMessage(normalizedEvent.data);
      }

      handlers?.onmessage?.(event);
    },
    [handlers, onError, onMessage],
  );

  const onerror: EventSourceHandlers['onerror'] = useCallback(
    (error: Error) => {
      setError(error.message || 'An error occurred');
      onError?.();
      handlers?.onerror?.(error);
    },
    [handlers, onError],
  );

  const onopen: EventSourceHandlers['onopen'] = useCallback(
    async (response: Response) => {
      setIsConnected(true);
      handlers?.onopen?.(response);
    },
    [handlers],
  );

  const onclose: EventSourceHandlers['onclose'] = useCallback(() => {
    setIsConnected(false);
    handlers?.onclose?.();
  }, [handlers]);

  const connect = useCallback(
    async (id: string): Promise<void> => {
      abort();

      abortController.current = new AbortController();
      try {
        await appSSE(
          id,
          {
            onmessage,
            onerror,
            onopen,
            onclose,
          },
          abortController.current,
        );
      } catch (err) {
        console.error(err);
        setError(
          err instanceof Error
            ? err.message
            : 'Unknown SSE error when connecting',
        );
      }
    },
    [abort, appSSE, onclose, onerror, onmessage, onopen],
  );

  return {
    connect,
    abort,
    error,
    progress,
    isConnected,
  };
};
