import {
  getAppLogs,
  getSerialMonitorLogs,
} from '@cloud-editor-mono/domain/src/services/services-by-app/app-lab';
import { AppDetailedInfo } from '@cloud-editor-mono/infrastructure';
import {
  AL_PYTHON_KEY,
  AL_SERIAL_MONITOR_KEY,
  AL_STARTUP_KEY,
  AppAction,
  AppLabActionStatus,
  AppTitleLogic,
  MultipleConsolePanelLogic,
  RuntimeActionsLogic,
  SwapRunningAppDialogLogic,
  TreeNode,
} from '@cloud-editor-mono/ui-components/lib/components-by-app/app-lab';
import { useCallback, useContext, useEffect, useState } from 'react';

import { RuntimeContext } from '../../providers/runtime/runtimeContext';
import { useBoardLifecycleStore } from '../../store/boards/boards';
import { useAppSSE } from './hooks/useAppSSE';
import { useAppWebSocket } from './hooks/useAppWebSocket';
import { checkIfHasIno, checkIfHasPython } from './utils/checkFileTypes';

export type UseAppDetailRuntimeLogic = (
  app?: AppDetailedInfo,
  fileTree?: TreeNode[],
  appTitleLogic?: () => ReturnType<AppTitleLogic>,
) => {
  swapRunningAppDialogLogic: SwapRunningAppDialogLogic;
  multipleConsolePanelLogic: MultipleConsolePanelLogic;
  runtimeActionsLogic: RuntimeActionsLogic;
};

export const useAppDetailRuntimeLogic: UseAppDetailRuntimeLogic = function (
  app?: AppDetailedInfo,
  fileTree?: TreeNode[],
  appTitleLogic?: () => ReturnType<AppTitleLogic>,
): ReturnType<UseAppDetailRuntimeLogic> {
  const [open, setOpen] = useState(false);

  const { selectedConnectedBoard: selectedBoard } = useBoardLifecycleStore();
  const {
    activeApp,
    currentAction,
    currentActionStatus,
    resetCurrentAction,
    consoleSources,
    consoleSourcesResetSubject: resetSource,
    runAction,
    stopAction,
    swapAction,
    consoleTabs,
    activeConsoleTab,
    setActiveConsoleTab,
    appendData,
    addConsoleSource,
    resetConsoleSources,
    consoleSourcesOwner,
  } = useContext(RuntimeContext);

  const runApp = useCallback((): void => {
    if (!app) return;
    runAction(app, setOpen);
  }, [app, runAction]);

  const stopApp = useCallback((): void => {
    if (!app) return;
    stopAction(app);
  }, [app, stopAction]);

  const {
    connect: getAppLogsStream,
    abort: getAppLogsAbort,
    //progress: getAppLogsProgress,
  } = useAppSSE({
    appSSE: getAppLogs,
    onMessage: (message) => {
      if (!message.id) return; //Some bricks are returning empty ids, this line prevents the logging of those
      appendData(message.id, message, true);
    },
  });

  const {
    connect: getSerialMonitorLogsStream,
    abort: getSerialMonitorLogsAbort,
    send: sendSerialMonitorLogsMessage,
  } = useAppWebSocket({
    appWebSocket: getSerialMonitorLogs,
    onMessage: (message) => {
      appendData(
        AL_SERIAL_MONITOR_KEY,
        {
          id: AL_SERIAL_MONITOR_KEY,
          message,
        },
        true,
      );
    },
  });

  useEffect(() => {
    const hasPython = fileTree ? checkIfHasPython(fileTree[0]) : false;
    const hasIno = fileTree ? checkIfHasIno(fileTree[0]) : false;

    //If there is a pending action, do not reset it.
    if (currentActionStatus !== AppLabActionStatus.Pending) {
      resetCurrentAction();
    }

    if (!currentAction) {
      resetConsoleSources([]);
    } else {
      resetConsoleSources([AL_STARTUP_KEY]);
      setActiveConsoleTab(AL_STARTUP_KEY);
    }

    if (app?.status === 'running') {
      if (hasPython) {
        setActiveConsoleTab(AL_PYTHON_KEY);
        addConsoleSource(AL_PYTHON_KEY);
        getAppLogsStream(app.id);
      }

      if (hasIno) {
        setActiveConsoleTab(AL_SERIAL_MONITOR_KEY);
        addConsoleSource(AL_SERIAL_MONITOR_KEY);
        //TODO!: getSerialMonitorLogsStream will not retrieve past messages. A way to save them is needed.
        getSerialMonitorLogsStream();
      }
    }

    return () => {
      getAppLogsAbort();
      getSerialMonitorLogsAbort();
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [app?.id, app?.status, fileTree]);

  const isActiveApp =
    activeApp?.id === app?.id ||
    consoleSourcesOwner === app?.id ||
    app?.status === 'running';

  const swapRunningAppDialogLogic: SwapRunningAppDialogLogic = useCallback(
    () => ({
      open,
      setOpen,
      handleSwap: (): void => {
        if (!app) return;
        swapAction(app);
        setOpen(false);
      },
    }),
    [app, open, swapAction],
  );

  const multipleConsolePanelLogic: MultipleConsolePanelLogic = useCallback(
    () => ({
      showLogs: isActiveApp,
      consoleTabs,
      consoleSources,
      activeTab: activeConsoleTab,
      setActiveTab: setActiveConsoleTab,
      resetSource: resetSource,
      onMessageSend: sendSerialMonitorLogsMessage,
      selectedBoard,
    }),
    [
      activeConsoleTab,
      consoleSources,
      consoleTabs,
      isActiveApp,
      resetSource,
      selectedBoard,
      sendSerialMonitorLogsMessage,
      setActiveConsoleTab,
    ],
  );

  const runtimeActionsLogic: RuntimeActionsLogic = useCallback(
    () => ({
      appId: app?.id || '',
      appName: app?.name || '',
      appStatus: app?.status || 'stopped',
      currentAction,
      currentActionStatus,
      runApp,
      stopApp,
      showStop: isActiveApp,
      isBannerEnabled: isActiveApp,
      duplicateApp:
        app?.example && appTitleLogic
          ? (): void => appTitleLogic().onAppAction(AppAction.Duplicate)
          : undefined,
    }),
    [
      app?.example,
      app?.id,
      app?.name,
      app?.status,
      appTitleLogic,
      currentAction,
      currentActionStatus,
      isActiveApp,
      runApp,
      stopApp,
    ],
  );

  return {
    swapRunningAppDialogLogic,
    multipleConsolePanelLogic,
    runtimeActionsLogic,
  };
};
